package ru.yandex.direct.logicprocessor.processors.bsexport.meaningfulgoals

import org.apache.commons.lang3.StringUtils
import org.springframework.stereotype.Service
import ru.yandex.direct.bstransport.yt.repository.meaningfulgoals.MeaningfulGoalsYtRepository
import ru.yandex.direct.bstransport.yt.repository.meaningfulgoals.container.MeaningfulGoalsExportInfo
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter
import ru.yandex.direct.ess.logicobjects.bsexport.meaningfulgoals.BsExportMeaningfulGoalsObject
import ru.yandex.direct.utils.CommonUtils.nvl
import ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8
import java.time.Clock

@Service
class BsExportMeaningfulGoalsService(
        private val bsOrderIdCalculator: BsOrderIdCalculator,
        private val clock: Clock = Clock.systemUTC(),
        private val ytRepository: MeaningfulGoalsYtRepository,
        private val logBsExportEssService: LogBsExportEssService) {

    companion object {
        private const val DATA_TYPE = "meaningful_goals"
    }

    fun processMeaningfulGoals(shard: Int, logicObjects: List<BsExportMeaningfulGoalsObject>) {
        if (logicObjects.isEmpty()) return

        val campaignIds = logicObjects.mapNotNull { it.campaignId }.toSet()
        val orderIdByCampaignId: Map<Long, Long> = bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, campaignIds)

        val updateTime = clock.instant().epochSecond
        val exportAndLogInfoCollection = logicObjects
                .filter { orderIdByCampaignId.containsKey(it.campaignId) } //когда кампания удалена из базы, order_id посчитать не получится
                .map { toExportAndLogInfo(it, orderIdByCampaignId, updateTime) }
                .map { it.copy(meaningfulGoalsInfo = getMeaningfulGoalsExportInfoWithDoubleValue(it.meaningfulGoalsInfo) ) }


        ytRepository.add(exportAndLogInfoCollection.map { it.meaningfulGoalsInfo })

        logBsExportEssService.logData(exportAndLogInfoCollection.map {
            LogBsExportEssData<MeaningfulGoalsExportInfo>()
                    .withCid(it.campaignId)
                    .withOrderId(it.meaningfulGoalsInfo.orderId)
                    .withData(it.meaningfulGoalsInfo)
        }, DATA_TYPE)
    }

    /**
     * Тут value в строковом представлении, в транспорт хотим продолжить отправлять число
     */
    private fun getMeaningfulGoalsExportInfoWithDoubleValue(meaningfulGoalsInfo: MeaningfulGoalsExportInfo): MeaningfulGoalsExportInfo {
        if (StringUtils.isEmpty(meaningfulGoalsInfo.meaningfulGoals)) {
            return meaningfulGoalsInfo
        }
        val meaningfulGoals = CampaignConverter.meaningfulGoalsFromDb(meaningfulGoalsInfo.meaningfulGoals)
        val convertedMeaningfulGoals = CampaignConverter.meaningfulGoalsToDb(meaningfulGoals, false);
        return meaningfulGoalsInfo.copy(meaningfulGoals = convertedMeaningfulGoals)
    }

    private fun toExportAndLogInfo(meaningfulGoalsObject: BsExportMeaningfulGoalsObject, orderIdByCampaignId: Map<Long, Long>, updateTime: Long) =
            ExportAndLogInfo(meaningfulGoalsObject.campaignId,
                    toMeaningfulGoalsExportInfo(orderIdByCampaignId, meaningfulGoalsObject, updateTime))

    private fun toMeaningfulGoalsExportInfo(orderIdByCampaignId: Map<Long, Long>,
                                            meaningfulGoalsObject: BsExportMeaningfulGoalsObject,
                                            updateTime: Long): MeaningfulGoalsExportInfo {
        val meaningfulGoals = nvl(meaningfulGoalsObject.meaningfulGoals, "")
        return MeaningfulGoalsExportInfo(
                orderIdByCampaignId.getValue(meaningfulGoalsObject.campaignId),
                getMd5HalfHashUtf8(meaningfulGoals).toLong(),
                updateTime,
                meaningfulGoals)
    }

    private data class ExportAndLogInfo(
            val campaignId: Long,
            val meaningfulGoalsInfo: MeaningfulGoalsExportInfo
    )
}
