package ru.yandex.direct.logicprocessor.processors.bsexport.metrikacounters

import ru.yandex.adv.direct.metrikacounters.CampaignMetrikaCounter
import ru.yandex.adv.direct.metrikacounters.MetrikaCounters
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.bstransport.yt.repository.metrikacounters.MetrikaCountersYtRepository
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter
import ru.yandex.direct.core.entity.campaign.repository.CampMetrikaCountersRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.ess.config.bsexport.metrikacounters.BsExportMetrikaCountersConfig
import ru.yandex.direct.ess.logicobjects.bsexport.metrikacounters.BsExportMetrikaCountersObject
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext
import ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8
import java.time.Clock


@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 20),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_1_NOT_READY, CheckTag.DIRECT_BS_EXPORT],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.CHAT_API_MONITORING],
        status = [JugglerStatus.OK, JugglerStatus.CRIT],
        method = [NotificationMethod.TELEGRAM]
    )]
)
@EssLogicProcessor(BsExportMetrikaCountersConfig::class)
class BsExportMetrikaCountersProcessor(
    essLogicProcessorContext: EssLogicProcessorContext,
    private val bsOrderIdCalculator: BsOrderIdCalculator,
    private val campMetrikaCountersRepository: CampMetrikaCountersRepository,
    private val metrikaCountersYtRepository: MetrikaCountersYtRepository,
    private val logBsExportEssService: LogBsExportEssService,
) : BaseLogicProcessor<BsExportMetrikaCountersObject>(essLogicProcessorContext) {

    companion object {
        const val LOG_DATA_TYPE = "metrika_counters"
    }

    data class CampaignWithMetrikaCountersData(
        val campaignId: Long,
        val metrikaCounters: MetrikaCounters,
    )

    override fun process(logicObjects: List<BsExportMetrikaCountersObject>) {
        if (logicObjects.isEmpty()) {
            return
        }

        val campaignIds = logicObjects.map { it.campaignId }
        val orderIdByCampaignId: Map<Long, Long> =
            bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, campaignIds)

        val campaignIdsWithOrderId = orderIdByCampaignId.keys

        val metrikaCounters: Map<Long, List<MetrikaCounter>> = campMetrikaCountersRepository
            .getMetrikaCounterByCid(shard, campaignIdsWithOrderId)
            .mapValues { (_, counters) -> counters.filter { !it.isDeleted } }
            .filterValues { counters -> counters.isNotEmpty() }

        val updateTime = Clock.systemUTC().instant().epochSecond
        val entries = metrikaCounters.map { (cid, counters) ->
            val orderId = orderIdByCampaignId[cid]!!
            CampaignWithMetrikaCountersData(cid, toMetrikaCountersExportInfo(orderId, counters, updateTime))
        }

        metrikaCountersYtRepository.modify(entries.map { it.metrikaCounters })

        logBsExportEssService.logData(entries.map {
            LogBsExportEssData<MetrikaCounters>()
                .withCid(it.campaignId)
                .withOrderId(it.metrikaCounters.orderId)
                .withData(it.metrikaCounters)
        }, LOG_DATA_TYPE)
    }

    private fun toMetrikaCountersExportInfo(
        orderId: Long,
        counters: List<MetrikaCounter>,
        updateTime: Long,
    ): MetrikaCounters {
        val counterIds = counters.map { it.id }

        val metrikaCountersHash = getCountersHash(counterIds)

        val metrikaCounters = counters.map { counter ->
            CampaignMetrikaCounter.newBuilder()
                .setId(counter.id)
                .build()
        }

        return MetrikaCounters.newBuilder().apply {
            setOrderId(orderId)
            setMetrikaCountersHash(metrikaCountersHash)
            addAllMetrikaCounters(metrikaCounters)
            setUpdateTime(updateTime)
        }.build()
    }

    /**
     * Аналогичный хеш вычисляется в транспорте кампании в перле.
     * Перед изменением стоит продумать как изменить хеш в обоих местах, не вызвав проблем в БК
     */
    private fun getCountersHash(counterIds: List<Long>): Long {
        val counters = counterIds.sorted().joinToString(",")
        return getMd5HalfHashUtf8(counters).toLong()
    }
}
