package ru.yandex.direct.logicprocessor.processors.bsexport.mobilegoals;

import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.util.Strings;
import org.springframework.stereotype.Service;

import ru.yandex.adv.direct.goals.GoalExternalTracker;
import ru.yandex.direct.bstransport.yt.repository.goals.MobileGoalsExternalTrackerYtRepository;
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData;
import ru.yandex.direct.common.log.service.LogBsExportEssService;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType;
import ru.yandex.direct.core.entity.mobileapp.model.MobileExternalTrackerEvent;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsExternalTrackerRepository;
import ru.yandex.direct.ess.logicobjects.bsexport.mobilegoals.BsExportMobileGoalsExternalTrackerObject;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class BsExportMobileGoalsExternalTrackerService {
    private static final String LOG_TYPE = "goal_external_tracker";
    private static final String LOG_DELETED_TYPE = "goal_external_tracker_deleted";
    private final MobileAppGoalsExternalTrackerRepository mobileAppGoalsExternalTrackerRepository;
    private final MobileAppRepository mobileAppRepository;
    private final MobileGoalsExternalTrackerYtRepository mobileGoalsExternalTrackerYtRepository;
    private final LogBsExportEssService logBsExportEssService;

    public BsExportMobileGoalsExternalTrackerService(
            MobileAppGoalsExternalTrackerRepository mobileAppGoalsExternalTrackerRepository,
            MobileAppRepository mobileAppRepository,
            MobileGoalsExternalTrackerYtRepository mobileGoalsExternalTrackerYtRepository,
            LogBsExportEssService logBsExportEssService) {
        this.mobileAppGoalsExternalTrackerRepository = mobileAppGoalsExternalTrackerRepository;
        this.mobileAppRepository = mobileAppRepository;
        this.mobileGoalsExternalTrackerYtRepository = mobileGoalsExternalTrackerYtRepository;
        this.logBsExportEssService = logBsExportEssService;
    }

    void processGoals(int shard, List<BsExportMobileGoalsExternalTrackerObject> objects) {
        List<Long> idsToUpdate = filterAndMapList(objects, o -> !o.isDeleted(),
                BsExportMobileGoalsExternalTrackerObject::getGoalId);
        updateGoals(shard, idsToUpdate);

        List<Long> idsToDelete = filterAndMapList(objects, BsExportMobileGoalsExternalTrackerObject::isDeleted,
                BsExportMobileGoalsExternalTrackerObject::getGoalId);
        deleteGoals(idsToDelete);
    }

    private void updateGoals(int shard, List<Long> goalIds) {
        List<MobileExternalTrackerEvent> goals = mobileAppGoalsExternalTrackerRepository.getEventsByIds(shard, goalIds);
        if (goals.isEmpty()) {
            return;
        }

        List<Long> mobileAppIds = mapList(goals, MobileExternalTrackerEvent::getMobileAppId);
        List<MobileApp> mobileApps = mobileAppRepository.getMobileApps(shard, null, mobileAppIds);
        Map<Long, MobileApp> mobileAppsByIds = listToMap(mobileApps, MobileApp::getId);

        List<GoalExternalTracker> goalsToModify = mapList(goals,
                goal -> GoalExternalTracker.newBuilder()
                        .setGoalId(goal.getId())
                        .setMobileAppId(goal.getMobileAppId())
                        .setEventName(goal.getEventName().toString())
                        .setBundleId(getBundleId(mobileAppsByIds, goal))
                        .setStoreType(MobileAppStoreType.toSource(mobileAppsByIds.get(goal.getMobileAppId()).getStoreType()).toString())
                        .build());

        logBsExportEssService.logData(mapList(goalsToModify, g -> new LogBsExportEssData<GoalExternalTracker>()
                .withData(g)), LOG_TYPE);
        mobileGoalsExternalTrackerYtRepository.modify(goalsToModify);
    }

    // Для ios поле bundle_id в таблице mobile_content заполнено, а для android берем его из поля store_content_id
    private String getBundleId(Map<Long, MobileApp> mobileAppsByIds, MobileExternalTrackerEvent goal) {
        MobileContent mobileContent = mobileAppsByIds.get(goal.getMobileAppId()).getMobileContent();
        return mobileContent == null ? "" : Strings.isBlank(mobileContent.getBundleId())
                ? mobileContent.getStoreContentId()
                : mobileContent.getBundleId();
    }

    private void deleteGoals(List<Long> goalIds) {
        if (goalIds.isEmpty()) {
            return;
        }
        logBsExportEssService.logData(mapList(goalIds, id -> new LogBsExportEssData<Long>()
                .withData(id)), LOG_DELETED_TYPE);
        mobileGoalsExternalTrackerYtRepository.deleteByIds(goalIds);
    }
}
