package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.expression.DeleteDirectMultipliersRow;
import ru.yandex.adv.direct.expression.DirectMultipliersRow;
import ru.yandex.adv.direct.expression.MultiplierAtom;
import ru.yandex.adv.direct.expression.MultiplierChangeRequest;
import ru.yandex.adv.direct.expression.multipler.type.MultiplierTypeEnum;
import ru.yandex.adv.direct.multipliers.Multiplier;
import ru.yandex.direct.bstransport.yt.repository.MultipliersYtRepository;
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData;
import ru.yandex.direct.common.log.service.LogBsExportEssService;
import ru.yandex.direct.core.bsexport.repository.BsExportMultipliersRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.AccessibleGoalChangedInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.BsExportMultipliersObject;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.DeleteInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.TimeTargetChangedInfo;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.CampaignWithTimeTarget;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierAndDeleteInfos;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler.BidModifierMultiplierHandler;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler.MultiplierHandler;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler.TimeMultiplierHandler;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Экспорт корректировок в общую базу в YT
 */
@Component
@ParametersAreNonnullByDefault
public class BsExportMultipliersService {
    private static final String LOG_TYPE = "multiplier";
    private static final Map<MultiplierType, List<BidModifierType>> BID_MODIFIER_TYPES_UNITED = Map.of(
            MultiplierType.DEVICE, List.of(
                    BidModifierType.DESKTOP_MULTIPLIER,
                    BidModifierType.MOBILE_MULTIPLIER,
                    BidModifierType.SMARTTV_MULTIPLIER,
                    BidModifierType.TABLET_MULTIPLIER,
                    BidModifierType.DESKTOP_ONLY_MULTIPLIER
            ),
            MultiplierType.INVENTORY, List.of(
                    BidModifierType.INVENTORY_MULTIPLIER,
                    BidModifierType.BANNER_TYPE_MULTIPLIER
            ),
            MultiplierType.RETARGETING, List.of(
                    BidModifierType.RETARGETING_MULTIPLIER,
                    BidModifierType.RETARGETING_FILTER
            )
    );
    private final org.slf4j.Logger logger = LoggerFactory.getLogger(BsExportMultipliersService.class);
    private final LogBsExportEssService logBsExportEssService;
    private final BidModifierRepository bidModifierRepository;
    private final BsExportMultipliersRepository bsExportMultipliersRepository;
    private final MultipliersYtRepository multipliersYtRepository;
    private final BsOrderIdCalculator bsOrderIdCalculator;
    private final Map<BidModifierType, BidModifierMultiplierHandler> multiplierHandlers;
    private final Map<MultiplierType, MultiplierTypeEnum> multiplierTypes;
    private final TimeMultiplierHandler timeMultiplierHandler;

    public BsExportMultipliersService(
            LogBsExportEssService logBsExportEssService,
            BidModifierRepository bidModifierRepository,
            BsExportMultipliersRepository bsExportMultipliersRepository,
            MultipliersYtRepository multipliersYtRepository,
            BsOrderIdCalculator bsOrderIdCalculator,
            List<BidModifierMultiplierHandler> bidModifierMultiplierHandlers,
            TimeMultiplierHandler timeMultiplierHandler) {
        this.logBsExportEssService = logBsExportEssService;
        this.bidModifierRepository = bidModifierRepository;
        this.bsExportMultipliersRepository = bsExportMultipliersRepository;
        this.multipliersYtRepository = multipliersYtRepository;
        this.bsOrderIdCalculator = bsOrderIdCalculator;

        this.multiplierHandlers = StreamEx.of(bidModifierMultiplierHandlers)
                .mapToEntry(BidModifierMultiplierHandler::getBidModifierTypes, Function.identity())
                .flatMapKeys(Collection::stream)
                .toMap();
        this.timeMultiplierHandler = timeMultiplierHandler;

        this.multiplierTypes = StreamEx.of(bidModifierMultiplierHandlers)
                .select(MultiplierHandler.class)
                .append(timeMultiplierHandler)
                .mapToEntry(MultiplierHandler::getMultiplierType, MultiplierHandler::getExportMultiplierType)
                .toMap();
    }

    public Map<Long, List<Multiplier>> getAdGroupMultipliers(int shard, Collection<Long> pids) {
        var ids = EntryStream.of(bidModifierRepository.getBidModifierIdsByAdGroupIds(shard, pids))
                .values()
                .toFlatList(Function.identity());
        var bidModifierKeys = bidModifierRepository.getBidModifierKeysByIds(shard, ids).values();
        var adGroupMultipliers = processMultipliersByKey(shard, bidModifierKeys)
                .getMultiplierInfos();

        return convertMultiplierInfoToMultipliers(adGroupMultipliers.stream(), MultiplierInfo::getAdGroupId);
    }

    public Map<Long, List<Multiplier>> getCampaignMultipliers(int shard, Collection<Long> cids) {
        var timeTargetMultipliers = processCampaignTimeTargetChanges(shard, cids)
                .getMultiplierInfos();

        // getBidModifierIdsByCampaignIds ищет только те корректировки, где pid == null
        var ids = EntryStream.of(bidModifierRepository.getBidModifierIdsByCampaignIds(shard, cids))
                .values()
                .toFlatList(Function.identity());
        var bidModifierKeys = bidModifierRepository.getBidModifierKeysByIds(shard, ids).values();
        var otherMultipliers = processMultipliersByKey(shard, bidModifierKeys)
                .getMultiplierInfos();

        var allMultipliers = StreamEx.of(timeTargetMultipliers).append(otherMultipliers);
        return convertMultiplierInfoToMultipliers(allMultipliers, MultiplierInfo::getCampaignId);
    }

    private Map<Long, List<Multiplier>> convertMultiplierInfoToMultipliers(Stream<MultiplierInfo> adGroupMultipliers,
                                                                           Function<MultiplierInfo, Long> classifier) {
        Collector<MultiplierInfo, ?, List<Multiplier>> collector = Collectors.flatMapping(multiplierInfo -> {
            if (!multiplierInfo.getEnabled()) {
                return Stream.empty();
            }

            List<MultiplierAtom> multipliers = nvl(multiplierInfo.getMultipliers(), Collections.emptyList());
            var multiplierType = multiplierTypes.get(multiplierInfo.getMultiplierType());
            return multipliers
                    .stream()
                    .map(multiplier -> MultiplierUtilsKt.toExpression2Format(multiplier, multiplierType));
        }, Collectors.toList());

        return StreamEx.of(adGroupMultipliers)
                .groupingBy(classifier, collector);
    }

    public void updateMultipliers(int shard, List<BsExportMultipliersObject> logicObjects) {
        logger.debug("Processing objects for (shard: {}): {}", shard, logicObjects);

        List<Long> toUpsertIds = new ArrayList<>();
        List<DeleteInfo> toDelete = new ArrayList<>();
        List<AccessibleGoalChangedInfo> toCheckAccessibleGoals = new ArrayList<>();
        List<TimeTargetChangedInfo> timeTargetChanges = new ArrayList<>();
        for (BsExportMultipliersObject logicObject : logicObjects) {
            if (logicObject.getDeleteInfo() != null) {
                toDelete.add(logicObject.getDeleteInfo());
            } else if (logicObject.getUpsertInfo() != null) {
                toUpsertIds.add(logicObject.getUpsertInfo().getHierarchicalMultiplierId());
            } else if (logicObject.getAccessibleGoalChangedInfo() != null) {
                toCheckAccessibleGoals.add(logicObject.getAccessibleGoalChangedInfo());
            } else {
                timeTargetChanges.add(checkNotNull(logicObject.getTimeTargetChangedInfo()));
            }
        }

        List<Long> multipliersFoundFromRetargetingCondition = convertToMultiplierIds(shard, toCheckAccessibleGoals);
        // добавляем события обновления ретаргетинговых корректировок в общий список, чтобы они перечитались из базы
        // с учётом видимости целей
        toUpsertIds.addAll(multipliersFoundFromRetargetingCondition);

        List<MultiplierInfo> multiplierInfos = new ArrayList<>();
        MultiplierAndDeleteInfos upsertProcessingResult = processUpsert(shard, toUpsertIds, toDelete);
        // после обработки событий изменения и вставки может оказаться,
        // что корректировку нужно удалить вместо изменения
        // например, такая ситуация возникает, когда в результате изменения ретаргетинговой корректировки
        // все значения корректировки оказались на основе условий с недоступными целями
        applyResults(upsertProcessingResult, multiplierInfos, toDelete);


        MultiplierAndDeleteInfos timeTargetProcessingResult = processTimeTargetChanges(shard, timeTargetChanges);
        applyResults(timeTargetProcessingResult, multiplierInfos, toDelete);

        Map<Long, Long> orderIdForCampaigns = obtainOrderIds(shard, toDelete, multiplierInfos);

        Stream<RequestWithExtra> upsertRequests = streamOfUpsertRequests(multiplierInfos, orderIdForCampaigns);
        Stream<RequestWithExtra> deleteRequests = streamOfDeleteRequests(toDelete, orderIdForCampaigns);

        List<RequestWithExtra> requests = Stream.concat(deleteRequests, upsertRequests).collect(Collectors.toList());
        logMultiplierExport(requests);

        List<MultiplierChangeRequest> changeRequests = requests.stream()
                .map(r -> r.request)
                .collect(Collectors.toList());
        multipliersYtRepository.changeMultipliers(changeRequests);
    }

    private void applyResults(MultiplierAndDeleteInfos multiplierAndDeleteInfos,
                              List<MultiplierInfo> multiplierInfos,
                              List<DeleteInfo> toDelete) {
        multiplierInfos.addAll(multiplierAndDeleteInfos.getMultiplierInfos());
        toDelete.addAll(multiplierAndDeleteInfos.getDeleteInfos());
    }

    private MultiplierAndDeleteInfos processTimeTargetChanges(
            int shard, List<TimeTargetChangedInfo> timeTargetChanges) {
        Set<Long> campaignIds = StreamEx.of(timeTargetChanges)
                .map(TimeTargetChangedInfo::getCampaignId)
                .nonNull()
                .toSet();
        return processCampaignTimeTargetChanges(shard, campaignIds);
    }

    private MultiplierAndDeleteInfos processCampaignTimeTargetChanges(int shard, Collection<Long> cids) {
        Map<Long, TimeTarget> campaignsTimeTarget =
                bsExportMultipliersRepository.getCampaignsTimeTargetWithoutAutobudget(shard, cids);

        List<CampaignWithTimeTarget> campaignWithTimeTargets = StreamEx.of(cids)
                .mapToEntry(Function.identity(), campaignsTimeTarget::get)
                .mapKeyValue(CampaignWithTimeTarget::new)
                .toList();

        return timeMultiplierHandler.handle(shard, campaignWithTimeTargets);
    }

    private MultiplierAndDeleteInfos processUpsert(int shard, List<Long> toUpsertIds, List<DeleteInfo> toDelete) {
        var bidModifierKeys = bidModifierRepository.getBidModifierKeysByIds(shard, toUpsertIds).values();

        // mobile и desktop корректировки логически объединены в псевдотип device, поэтому, при изменении или удалении
        // одного из них, нужно подгрузить и экспортировать дополняющий тип вместе + то же самое с inventory и retargeting
        var expandedKeys = EntryStream.of(BID_MODIFIER_TYPES_UNITED)
                .mapKeyValue((multiplierType,  modifierTypes) ->
                        expandComplementaryMultipliers(bidModifierKeys, toDelete, multiplierType, modifierTypes))
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        if (expandedKeys.isEmpty()) {
            return new MultiplierAndDeleteInfos(List.of(), List.of());
        }

        return processMultipliersByKey(shard, expandedKeys);
    }

    private MultiplierAndDeleteInfos processMultipliersByKey(int shard, Collection<BidModifierKey> bidModifierKeys) {
        var bidModifiers = bidModifierRepository
                .getBidModifiersByKeys(shard, bidModifierKeys)
                .values();

        // группировка корректировок по хэндлерам, учитывая поддерживаемые типы
        List<MultiplierInfo> multiplierInfos = new ArrayList<>();
        List<DeleteInfo> deleteInfos = new ArrayList<>();

        // группировка корректировок по хэндлерам, учитывая поддерживаемые типы
        Map<BidModifierMultiplierHandler, List<BidModifier>> groups = StreamEx.of(bidModifiers)
                .mapToEntry(BidModifier::getType, Function.identity())
                .filterKeys(multiplierHandlers::containsKey)
                .mapKeys(multiplierHandlers::get)
                .grouping(IdentityHashMap::new);

        EntryStream.of(groups)
                .mapKeyValue((mh, objects) -> mh.handle(shard, objects))
                .forEach(multiplierAndDeleteInfos ->
                        applyResults(multiplierAndDeleteInfos, multiplierInfos, deleteInfos));
        return new MultiplierAndDeleteInfos(multiplierInfos, deleteInfos);
    }

    private Stream<RequestWithExtra> streamOfUpsertRequests(
            List<MultiplierInfo> multiplierInfos, Map<Long, Long> orderIdForCampaigns) {
        return StreamEx.of(multiplierInfos)
                .filter(o -> orderIdForCampaigns.containsKey(o.getCampaignId()))
                .map(o -> createUpsertRequestWithExtra(o, orderIdForCampaigns));
    }

    private Stream<RequestWithExtra> streamOfDeleteRequests(
            List<DeleteInfo> toDelete, Map<Long, Long> orderIdForCampaigns) {
        return StreamEx.of(toDelete)
                .distinct()
                .filter(o -> orderIdForCampaigns.containsKey(o.getCampaignId()))
                .map(o -> createDeleteRequestWithExtra(o, orderIdForCampaigns));
    }

    private RequestWithExtra createUpsertRequestWithExtra(
            MultiplierInfo multiplierInfo, Map<Long, Long> orderIdForCampaigns) {
        var innerRequest = DirectMultipliersRow.newBuilder()
                .setOrderID(orderIdForCampaigns.get(multiplierInfo.getCampaignId()))
                .setAdGroupID(nvl(multiplierInfo.getAdGroupId(), 0L))
                .setType(multiplierTypes.get(multiplierInfo.getMultiplierType()))
                .setIsEnabled(multiplierInfo.getEnabled())
                .addAllMultipliers(multiplierInfo.getMultipliers())
                .build();
        var request = MultiplierChangeRequest.newBuilder().setUpsertRequest(innerRequest).build();
        return new RequestWithExtra(request, multiplierInfo.getCampaignId(), multiplierInfo.getAdGroupId());
    }

    private RequestWithExtra createDeleteRequestWithExtra(DeleteInfo deleteInfo, Map<Long, Long> orderIdForCampaigns) {
        var innerRequest = DeleteDirectMultipliersRow.newBuilder()
                .setOrderID(orderIdForCampaigns.get(deleteInfo.getCampaignId()))
                .setAdGroupID(nvl(deleteInfo.getAdGroupId(), 0L))
                .setType(multiplierTypes.get(deleteInfo.getMultiplierType()))
                .build();
        var request = MultiplierChangeRequest.newBuilder().setDeleteRequest(innerRequest).build();
        return new RequestWithExtra(request, deleteInfo.getCampaignId(), deleteInfo.getAdGroupId());
    }

    private Map<Long, Long> obtainOrderIds(int shard, List<DeleteInfo> toDelete, List<MultiplierInfo> multiplierInfos) {
        Set<Long> mentionedCampaignIds = Sets.union(
                listToSet(toDelete, (DeleteInfo::getCampaignId)),
                listToSet(multiplierInfos, MultiplierInfo::getCampaignId));
        return bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, mentionedCampaignIds);
    }

    /**
     * Дополняет список ключей для типов корретировок, объединенных в группы.
     * Например, для корректировки с типом retargeting также надо обновлять retargeting_filter, чтобы не потерять при транспорте.
     * @param multiplierType псевдотип, под которым типы корректировок объединены и передаются вместе
     * @param modifierTypes группа типов, объединенных в multiplierType
     */
    private Set<BidModifierKey> expandComplementaryMultipliers(Collection<BidModifierKey> bidModifierKeys,
                                                               List<DeleteInfo> toDelete,
                                                               MultiplierType multiplierType,
                                                               List<BidModifierType> modifierTypes) {
        // Если удалён какой-то тип корректировок в группе, то стоит проверить, а не
        // остались ли другие типы корректировок в группе, и если остались -- сделать изменение строки в экспортируемой таблице
        StreamEx<BidModifierKey> maybeDeleteMultipliers = StreamEx.of(toDelete)
                .filter(i -> i.getMultiplierType() == multiplierType)
                .flatCollection(i ->
                        mapList(modifierTypes, type -> new BidModifierKey(i.getCampaignId(), i.getAdGroupId(), type))
                );
        return StreamEx.of(bidModifierKeys)
                .flatMap(k -> {
                    if (modifierTypes.contains(k.getType())) {
                        return StreamEx.of(modifierTypes)
                                .map(type -> new BidModifierKey(k.getCampaignId(), k.getAdGroupId(), type));
                    } else {
                        return Stream.of(k);
                    }
                })
                .append(maybeDeleteMultipliers)
                .toSet();
    }

    private void logMultiplierExport(List<RequestWithExtra> requestWithExtraList) {
        var logs = requestWithExtraList.stream()
                .map(requestWithExtra -> {
                    var request = requestWithExtra.request;
                    long orderId = request.hasUpsertRequest() ?
                            request.getUpsertRequest().getOrderID() :
                            request.getDeleteRequest().getOrderID();

                    var logEntry = new LogBsExportEssData<MultiplierChangeRequest>()
                            .withCid(requestWithExtra.campaignId)
                            .withOrderId(orderId)
                            .withData(request);
                    if (requestWithExtra.adGroupId != null) {
                        logEntry.withPid(requestWithExtra.adGroupId);
                    }
                    return logEntry;
                })
                .collect(Collectors.toList());
        logBsExportEssService.logData(logs, LOG_TYPE);
    }

    private List<Long> convertToMultiplierIds(int shard, List<AccessibleGoalChangedInfo> toCheckAccessibleGoals) {
        if (toCheckAccessibleGoals.isEmpty()) {
            return List.of();
        }
        // AccessibleGoalChangedInfo события об изменении доступности целей, входящих в условие ретаргетинга
        // при таком изменении нужно переэкспортировть ретаргетинговые корректировки и корректировки на AБ-сегменты,
        // т.к. их экспорт зависит от доступности целей
        Set<Long> retargetingConditionIds = StreamEx.of(toCheckAccessibleGoals)
                .map(AccessibleGoalChangedInfo::getRetargetingConditionId)
                .toSet();
        // получим корректировки, где используются эти условия ретаргетинга
        // (retargeting_multiplier_values и ab_segments_multiplier_values)
        return bsExportMultipliersRepository.getMultiplierIdsByRetargetingConditionIds(
                shard, retargetingConditionIds);
    }

    // Класс для сохранения доп. информации для логирования
    private static class RequestWithExtra {
        private final MultiplierChangeRequest request;
        private final Long campaignId;
        @Nullable
        private final Long adGroupId;

        public RequestWithExtra(MultiplierChangeRequest request, Long campaignId, @Nullable Long adGroupId) {
            this.request = request;
            this.campaignId = campaignId;
            this.adGroupId = adGroupId;
        }
    }

}
