package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;

import ru.yandex.adv.direct.expression.MultiplierAtom;
import ru.yandex.adv.direct.expression.TargetingExpression;
import ru.yandex.adv.direct.expression.TargetingExpressionAtom;
import ru.yandex.adv.direct.expression.keywords.KeywordEnum;
import ru.yandex.adv.direct.expression.operations.OperationEnum;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionLiteral;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionOperator;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionParameter;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo;

import static java.util.Map.entry;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public abstract class BaseExpressionMultiplierHandler extends BaseMultiplierHandler {
    static final Map<BidModifierExpressionParameter, KeywordEnum> KEYWORD_MAP = Map.ofEntries(
            entry(BidModifierExpressionParameter.TRAFFIC_JAM, KeywordEnum.TrafficJam),
            entry(BidModifierExpressionParameter.VIDEO_CONTENT_DURATION, KeywordEnum.VideoContentDuration),
            entry(BidModifierExpressionParameter.PREC_STRENGTH, KeywordEnum.PrecipitationStrength),
            entry(BidModifierExpressionParameter.CLOUDNESS, KeywordEnum.Cloudness),
            entry(BidModifierExpressionParameter.TEMP, KeywordEnum.Temperature),
            entry(BidModifierExpressionParameter.PRISMA_INCOME_GRADE, KeywordEnum.PrismaIncomeGrade)
    );

    static final Map<BidModifierExpressionOperator, OperationEnum> OPERATION_MAP = Map.ofEntries(
            entry(BidModifierExpressionOperator.GT, OperationEnum.Greater),
            entry(BidModifierExpressionOperator.GE, OperationEnum.GreaterOrEqual),
            entry(BidModifierExpressionOperator.LT, OperationEnum.Less),
            entry(BidModifierExpressionOperator.LE, OperationEnum.LessOrEqual),
            entry(BidModifierExpressionOperator.EQ, OperationEnum.Equal),
            entry(BidModifierExpressionOperator.NE, OperationEnum.NotEqual),
            entry(BidModifierExpressionOperator.MATCH_GOAL_CONTEXT, OperationEnum.MatchGoalContext)
    );

    @Override
    public List<MultiplierInfo> convert(Collection<? extends BidModifier> bidModifiers) {
        return StreamEx.of(bidModifiers)
                .select(BidModifierExpression.class)
                .map(this::toMultiplierInfo)
                .toList();
    }

    private MultiplierInfo toMultiplierInfo(BidModifierExpression modifier) {
        return new MultiplierInfo(
                getMultiplierType(), modifier.getCampaignId(), modifier.getAdGroupId(), modifier.getEnabled(),
                mapList(modifier.getExpressionAdjustments(), this::toMultiplierAtom)
        );
    }

    private MultiplierAtom toMultiplierAtom(BidModifierExpressionAdjustment adjustment) {
        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(toCondition(adjustment.getCondition()))
                .build();
    }

    private TargetingExpression toCondition(List<List<BidModifierExpressionLiteral>> expression) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        for (List<BidModifierExpressionLiteral> disjunction : expression) {
            TargetingExpression.Disjunction.Builder disjunctionBuilder = TargetingExpression.Disjunction.newBuilder();
            for (BidModifierExpressionLiteral expressionLiteral : disjunction) {
                disjunctionBuilder.addOR(toTargetingExpressionAtom(expressionLiteral));
            }
            targetingExpressionBuilder.addAND(disjunctionBuilder.build());
        }
        return targetingExpressionBuilder.build();
    }

    private TargetingExpressionAtom toTargetingExpressionAtom(BidModifierExpressionLiteral expressionLiteral) {
        String value = expressionLiteral.getValueString();
        if (value == null) {
            value = expressionLiteral.getValueInteger().toString();
        }
        return TargetingExpressionAtom.newBuilder()
                .setKeyword(toKeyword(expressionLiteral.getParameter()))
                .setOperation(toOperation(expressionLiteral.getOperation()))
                .setValue(value)
                .build();
    }

    private KeywordEnum toKeyword(BidModifierExpressionParameter expressionParameter) {
        if (KEYWORD_MAP.containsKey(expressionParameter)) {
            return KEYWORD_MAP.get(expressionParameter);
        } else {
            throw new IllegalArgumentException(
                    String.format("Unknown expression parameter %s", expressionParameter));
        }
    }

    private OperationEnum toOperation(BidModifierExpressionOperator operationType) {
        if (OPERATION_MAP.containsKey(operationType)) {
            return OPERATION_MAP.get(operationType);
        } else {
            throw new IllegalArgumentException(String.format("Unsupported operation type %s", operationType));
        }
    }
}
