package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.expression.MultiplierAtom;
import ru.yandex.adv.direct.expression.TargetingExpression;
import ru.yandex.adv.direct.expression.TargetingExpressionAtom;
import ru.yandex.adv.direct.expression.keywords.KeywordEnum;
import ru.yandex.adv.direct.expression.multipler.type.MultiplierTypeEnum;
import ru.yandex.adv.direct.expression.operations.OperationEnum;
import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo;

import static java.util.Map.entry;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DemographyMultiplierHandler extends BaseMultiplierHandler {
    static final Map<GenderType, String> GENDER_TYPE_MAP = Map.ofEntries(
            entry(GenderType.FEMALE, "1"),
            entry(GenderType.MALE, "0")
    );

    static final Map<AgeType, String> AGE_TYPE_MAP = Map.ofEntries(
            entry(AgeType._0_17, "0"),
            entry(AgeType._18_24, "1"),
            entry(AgeType._25_34, "2"),
            entry(AgeType._35_44, "3"),
            entry(AgeType._45_54, "4"),
            entry(AgeType._55_, "5")
    );

    @Override
    public MultiplierType getMultiplierType() {
        return MultiplierType.DEMOGRAPHY;
    }

    @Override
    public MultiplierTypeEnum getExportMultiplierType() {
        return MultiplierTypeEnum.Demography;
    }

    @Override
    public List<MultiplierInfo> convert(Collection<? extends BidModifier> bidModifiers) {
        return StreamEx.of(bidModifiers)
                .select(BidModifierDemographics.class)
                .map(this::toMultiplierInfo)
                .toList();
    }

    private MultiplierInfo toMultiplierInfo(BidModifierDemographics modifier) {
        return new MultiplierInfo(
                MultiplierType.DEMOGRAPHY, modifier.getCampaignId(), modifier.getAdGroupId(), modifier.getEnabled(),
                mapList(modifier.getDemographicsAdjustments(), this::toMultiplierAtom)
        );
    }

    private MultiplierAtom toMultiplierAtom(BidModifierDemographicsAdjustment adjustment) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        if (adjustment.getGender() != null) {
            targetingExpressionBuilder.addAND(
                    TargetingExpression.Disjunction.newBuilder()
                            .addOR(
                                    TargetingExpressionAtom.newBuilder()
                                            .setKeyword(KeywordEnum.CryptaSocdemGender)
                                            .setOperation(OperationEnum.Equal)
                                            .setValue(convertGender(adjustment.getGender()))
                                            .build()
                            )
            );
        }
        if (adjustment.getAge() != null) {
            if (adjustment.getAge() == AgeType.UNKNOWN) {
                // Неизвестный возраст описывается через отрицание известных возрастов
                for (String ageValue : AGE_TYPE_MAP.values()) {
                    targetingExpressionBuilder.addAND(
                            TargetingExpression.Disjunction.newBuilder()
                                    .addOR(
                                            TargetingExpressionAtom.newBuilder()
                                                    .setKeyword(KeywordEnum.CryptaSocdemAge)
                                                    .setOperation(OperationEnum.NotEqual)
                                                    .setValue(ageValue)
                                                    .build()
                                    )
                    );
                }
            } else {
                TargetingExpression.Disjunction.Builder disjunctionBuilder =
                        TargetingExpression.Disjunction.newBuilder();
                for (String ageValue : convertAge(adjustment.getAge())) {
                    disjunctionBuilder.addOR(TargetingExpressionAtom.newBuilder()
                            .setKeyword(KeywordEnum.CryptaSocdemAge)
                            .setOperation(OperationEnum.Equal)
                            .setValue(ageValue)
                            .build()
                    );
                }
                targetingExpressionBuilder.addAND(disjunctionBuilder.build());
            }
        }
        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(targetingExpressionBuilder.build())
                .build();
    }

    private String convertGender(GenderType genderType) {
        if (GENDER_TYPE_MAP.containsKey(genderType)) {
            return GENDER_TYPE_MAP.get(genderType);
        } else {
            throw new IllegalArgumentException(
                    String.format("Unknown gender %s", genderType));
        }
    }

    private List<String> convertAge(AgeType ageType) {
        if (ageType == AgeType._45_) {
            return List.of(AGE_TYPE_MAP.get(AgeType._45_54), AGE_TYPE_MAP.get(AgeType._55_));
        }
        if (AGE_TYPE_MAP.containsKey(ageType)) {
            return List.of(AGE_TYPE_MAP.get(ageType));
        } else {
            throw new IllegalArgumentException(
                    String.format("Unknown age %s", ageType));
        }
    }
}
