package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.expression.MultiplierAtom;
import ru.yandex.adv.direct.expression.TargetingExpression;
import ru.yandex.adv.direct.expression.TargetingExpressionAtom;
import ru.yandex.adv.direct.expression.keywords.KeywordEnum;
import ru.yandex.adv.direct.expression.multipler.type.MultiplierTypeEnum;
import ru.yandex.adv.direct.expression.operations.OperationEnum;
import ru.yandex.direct.core.bsexport.repository.BsExportMultipliersRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnlyAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTVAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.OsType;
import ru.yandex.direct.core.entity.bidmodifier.TabletOsType;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.DeleteInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierAndDeleteInfos;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class DeviceMultiplierHandler implements BidModifierMultiplierHandler {
    /**
     * see:
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/tools/logs_docs/test/canondata/exectest.run_logs_docs_/logs_docs.out.md?rev=7322093#enum-detaileddevicetypeenum-2">DetailedDeviceTypeEnum</a>
     */
    private static final String DETAILED_DEVICE_TYPE_ANDROID = "2";
    private static final String DETAILED_DEVICE_TYPE_IOS = "3";
    @Autowired
    private final BsExportMultipliersRepository bsExportMultipliersRepository;

    public DeviceMultiplierHandler(BsExportMultipliersRepository bsExportMultipliersRepository) {
        this.bsExportMultipliersRepository = bsExportMultipliersRepository;
    }

    @Override
    public MultiplierType getMultiplierType() {
        return MultiplierType.DEVICE;
    }

    @Override
    public MultiplierTypeEnum getExportMultiplierType() {
        return MultiplierTypeEnum.DeviceType;
    }

    @Override
    public MultiplierAndDeleteInfos handle(int shard, Collection<? extends BidModifier> bidModifiers) {
        Map<DeviceTypeMultiplierKey, ? extends List<? extends BidModifier>> groups = StreamEx.of(bidModifiers)
                .mapToEntry(m -> new DeviceTypeMultiplierKey(m.getCampaignId(), m.getAdGroupId()), Function.identity())
                .grouping();

        List<MultiplierInfo> multiplierInfos = new ArrayList<>();
        List<DeleteInfo> deleteInfos = new ArrayList<>();
        Set<Long> campaignIds = listToSet(bidModifiers, BidModifier::getCampaignId);
        Map<Long, Boolean> smartTVEnableByCampaignIds =
                bsExportMultipliersRepository.getSmartTVEnableByCampaignIds(shard, campaignIds);

        EntryStream.of(groups)
                .forKeyValue((key, modifiers) -> {
                    if (modifiers.stream().noneMatch(BidModifier::getEnabled)) {
                        // не выгружаем выключенные корректировки
                        deleteInfos.add(new DeleteInfo(MultiplierType.DEVICE, key.campaignId, key.adGroupId));
                    } else {
                        multiplierInfos.add(toMultiplierInfo(key, modifiers,
                                smartTVEnableByCampaignIds.getOrDefault(key.campaignId, false)));
                    }
                });

        return new MultiplierAndDeleteInfos(multiplierInfos, deleteInfos);
    }

    private MultiplierInfo toMultiplierInfo(DeviceTypeMultiplierKey key, List<? extends BidModifier> modifiers,
                                            boolean isSmartTVEnable) {
        Optional<BidModifierMobile> mobile = StreamEx.of(modifiers)
                .select(BidModifierMobile.class)
                .filter(BidModifier::getEnabled)
                .findFirst();
        Optional<BidModifierDesktop> desktop = StreamEx.of(modifiers)
                .select(BidModifierDesktop.class)
                .filter(BidModifier::getEnabled)
                .findFirst();
        Optional<BidModifierDesktopOnly> desktopOnly = StreamEx.of(modifiers)
                .select(BidModifierDesktopOnly.class)
                .filter(BidModifier::getEnabled)
                .findFirst();
        Optional<BidModifierTablet> tablet = StreamEx.of(modifiers)
                .select(BidModifierTablet.class)
                .filter(BidModifier::getEnabled)
                .findFirst();
        Optional<BidModifierSmartTV> smartTV = StreamEx.of(modifiers)
                .select(BidModifierSmartTV.class)
                .filter(BidModifier::getEnabled)
                .findFirst();

        var multiplierAtoms= StreamEx.of(mobile.map(BidModifierMobile::getMobileAdjustment).map(adj -> toMultiplierAtom(adj, isSmartTVEnable)))
                .append(StreamEx.of(desktop.map(BidModifierDesktop::getDesktopAdjustment).map(this::toMultiplierAtom)))
                .append(StreamEx.of(desktopOnly.map(BidModifierDesktopOnly::getDesktopOnlyAdjustment).map(this::toMultiplierAtom)))
                .append(StreamEx.of(tablet.map(BidModifierTablet::getTabletAdjustment).map(this::toMultiplierAtom)));

        if (isSmartTVEnable) {
            multiplierAtoms = multiplierAtoms.append(StreamEx.of(smartTV.map(BidModifierSmartTV::getSmartTVAdjustment).map(this::toMultiplierAtom)));
        }

        return new MultiplierInfo(MultiplierType.DEVICE, key.campaignId, key.adGroupId, Boolean.TRUE, multiplierAtoms.toList());
    }


    private MultiplierAtom toMultiplierAtom(BidModifierMobileAdjustment adjustment, boolean isSmartTVEnabled) {
        OsType osType = adjustment.getOsType();
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.DeviceIsMobile)
                                        .setOperation(OperationEnum.Equal)
                                        .setValue("1")
                                        .build()
                        )
                        .build()
        );
        if (osType == OsType.ANDROID) {
            //Отправляем DeviceType = Android AND SmartTV = 0.
            targetingExpressionBuilder.addAND(
                    TargetingExpression.Disjunction.newBuilder()
                            .addOR(
                                    TargetingExpressionAtom.newBuilder()
                                            .setKeyword(KeywordEnum.DetailedDeviceType)
                                            .setOperation(OperationEnum.Equal)
                                            .setValue(DETAILED_DEVICE_TYPE_ANDROID) // DeviceType = Android
                                            .build()
                            )
                            .build()
            );
        } else if (osType == OsType.IOS) {
            targetingExpressionBuilder.addAND(
                    TargetingExpression.Disjunction.newBuilder()
                            .addOR(
                                    TargetingExpressionAtom.newBuilder()
                                            .setKeyword(KeywordEnum.DetailedDeviceType)
                                            .setOperation(OperationEnum.Equal)
                                            .setValue(DETAILED_DEVICE_TYPE_IOS)
                                            .build()
                            )
                            .build()
            );
        }
        if (osType != OsType.IOS && isSmartTVEnabled) {
            targetingExpressionBuilder.addAND(
                    TargetingExpression.Disjunction.newBuilder()
                            .addOR(
                                    TargetingExpressionAtom.newBuilder()
                                            .setKeyword(KeywordEnum.SmartTv)
                                            .setOperation(OperationEnum.Equal)
                                            .setValue("0") // SmartTV = 0
                                            .build()
                            )
                            .build()
            );
        }


        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(targetingExpressionBuilder.build())
                .build();
    }

    private MultiplierAtom toMultiplierAtom(BidModifierDesktopAdjustment adjustment) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.DeviceIsDesktop)
                                        .setOperation(OperationEnum.Equal)
                                        .setValue("1")
                                        .build()
                        )
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.DeviceIsTablet)
                                        .setOperation(OperationEnum.Equal)
                                        .setValue("1")
                                        .build()
                        )
                        .build()
        );
        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(targetingExpressionBuilder.build())
                .build();
    }

    private MultiplierAtom toMultiplierAtom(BidModifierDesktopOnlyAdjustment adjustment) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.DeviceIsDesktop)
                                        .setOperation(OperationEnum.Equal)
                                        .setValue("1")
                                        .build()
                        )
                        .build()
        );
        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(targetingExpressionBuilder.build())
                .build();
    }

    private MultiplierAtom toMultiplierAtom(BidModifierTabletAdjustment adjustment) {
        var osType = adjustment.getOsType();

        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.DeviceIsTablet)
                                        .setOperation(OperationEnum.Equal)
                                        .setValue("1")
                                        .build()
                        )
                        .build()
        );


        if (osType == TabletOsType.ANDROID) {
            //Отправляем DeviceType = Android AND SmartTV = 0.
            targetingExpressionBuilder.addAND(
                    TargetingExpression.Disjunction.newBuilder()
                            .addOR(
                                    TargetingExpressionAtom.newBuilder()
                                            .setKeyword(KeywordEnum.DetailedDeviceType)
                                            .setOperation(OperationEnum.Equal)
                                            .setValue(DETAILED_DEVICE_TYPE_ANDROID) // DeviceType = Android
                                            .build()
                            )
                            .build()
            );
        } else if (osType == TabletOsType.IOS) {
            targetingExpressionBuilder.addAND(
                    TargetingExpression.Disjunction.newBuilder()
                            .addOR(
                                    TargetingExpressionAtom.newBuilder()
                                            .setKeyword(KeywordEnum.DetailedDeviceType)
                                            .setOperation(OperationEnum.Equal)
                                            .setValue(DETAILED_DEVICE_TYPE_IOS)
                                            .build()
                            )
                            .build()
            );
        }

        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(targetingExpressionBuilder.build())
                .build();
    }

    private MultiplierAtom toMultiplierAtom(BidModifierSmartTVAdjustment adjustment) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.DeviceIsMobile)
                                        .setOperation(OperationEnum.Equal)
                                        .setValue("1")
                                        .build()
                        )
                        .build()
        );
        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.SmartTv)
                                        .setOperation(OperationEnum.Equal)
                                        .setValue("1") //SmartTV = 1
                                        .build()
                        )
                        .build()
        );
        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(targetingExpressionBuilder.build())
                .build();
    }

    private static class DeviceTypeMultiplierKey {
        private final long campaignId;
        @Nullable
        private final Long adGroupId;

        public DeviceTypeMultiplierKey(long campaignId, @Nullable Long adGroupId) {
            this.campaignId = campaignId;
            this.adGroupId = adGroupId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DeviceTypeMultiplierKey that = (DeviceTypeMultiplierKey) o;
            return campaignId == that.campaignId &&
                    Objects.equals(adGroupId, that.adGroupId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(campaignId, adGroupId);
        }
    }
}
