package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.expression.CustomExpression
import ru.yandex.adv.direct.expression.InventoryTypeEnum
import ru.yandex.adv.direct.expression.MultiplierAtom
import ru.yandex.adv.direct.expression.multipler.type.MultiplierTypeEnum
import ru.yandex.direct.core.entity.bidmodifier.BidModifier
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment
import ru.yandex.direct.core.entity.bidmodifier.InventoryType
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.DeleteInfo
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType.INVENTORY
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierAndDeleteInfos
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo

@Component
class InventoryTypeHandler : BidModifierMultiplierHandler {
    companion object {
        val INVENTORY_TYPE_MAPPING: Map<InventoryType, InventoryTypeEnum> = mapOf(
            InventoryType.INSTREAM_WEB to InventoryTypeEnum.VideoInStream,
            InventoryType.INPAGE to InventoryTypeEnum.VideoInPage,
            InventoryType.INAPP to InventoryTypeEnum.VideoInApp,
            InventoryType.INBANNER to InventoryTypeEnum.VideoInBanner,
            InventoryType.REWARDED to InventoryTypeEnum.VideoRewarded,
            InventoryType.PREROLL to InventoryTypeEnum.VideoPreroll,
            InventoryType.MIDROLL to InventoryTypeEnum.VideoMidroll,
            InventoryType.POSTROLL to InventoryTypeEnum.VideoPostroll,
            InventoryType.PAUSEROLL to InventoryTypeEnum.VideoPauseroll,
            InventoryType.OVERLAY to InventoryTypeEnum.VideoOverlay,
            InventoryType.POSTROLL_OVERLAY to InventoryTypeEnum.VideoPostrollOverlay,
            InventoryType.POSTROLL_WRAPPER to InventoryTypeEnum.VideoPostrollWrapper,
            InventoryType.INROLL_OVERLAY to InventoryTypeEnum.VideoInrollOverlay,
            InventoryType.INROLL to InventoryTypeEnum.VideoInroll,
            InventoryType.INTERSTITIAL to InventoryTypeEnum.VideoInterstitial,
            InventoryType.FULLSCREEN to InventoryTypeEnum.VideoFullscreen,
        )
    }

    override fun getMultiplierType() = INVENTORY

    override fun getExportMultiplierType() = MultiplierTypeEnum.InventoryType

    override fun handle(shard: Int, bidModifiers: Collection<BidModifier>): MultiplierAndDeleteInfos {
        // объединим корректировки по принадлежности к кампании или группе
        // в таком объединение может быть 1 или 2 элемента, 2 в случае,
        // когда на группе или кампании есть и BidModifierBannerType и BidModifierInventory
        // их нужно экспортировать вместе как одну корректировку InventoryType
        val joinedModifiers = bidModifiers.groupBy { InventoryTypeKey(it.campaignId, it.adGroupId) }
            .map { (key, modifiers) -> joinModifiers(key, modifiers) }

        val multiplierInfos = mutableListOf<MultiplierInfo>()
        val deleteInfos = mutableListOf<DeleteInfo>()

        joinedModifiers.forEach { joinedModifier ->
            if (joinedModifier.enabled) {
                multiplierInfos.add(toMultiplierInfo(joinedModifier))
            } else {
                deleteInfos.add(DeleteInfo(INVENTORY, joinedModifier.campaignId, joinedModifier.adGroupId))
            }
        }
        return MultiplierAndDeleteInfos(multiplierInfos, deleteInfos)
    }

    private fun joinModifiers(key: InventoryTypeKey, modifiers: List<BidModifier>): JoinedInventoryTypeBidModifier {
        modifiers.forEach {
            check(it is BidModifierBannerType || it is BidModifierInventory) {
                "Unexpected type ${modifiers.javaClass} of bid modifier ID=${it.id}"
            }
        }
        return JoinedInventoryTypeBidModifier(
            key,
            modifiers.filterIsInstance<BidModifierBannerType>().firstOrNull(),
            modifiers.filterIsInstance<BidModifierInventory>().firstOrNull()
        )
    }

    private fun toMultiplierInfo(joinedModifier: JoinedInventoryTypeBidModifier): MultiplierInfo {
        val enabledBannerTypeMultiplierAtoms = sequenceOf(joinedModifier.bannerType)
            .filterNotNull()
            .filter { it.enabled }
            .flatMap { it.bannerTypeAdjustments }
            .map { toMultiplierAtom(it) }
            .toList()
        val enabledInventoryMultiplierAtoms = sequenceOf(joinedModifier.inventory)
            .filterNotNull()
            .filter { it.enabled }
            .flatMap { it.inventoryAdjustments }
            .map { toMultiplierAtom(it) }
            .toList()
        val defaultMultiplier = mutableListOf(createDefaultMultiplierAtom())
        return MultiplierInfo(
            INVENTORY, joinedModifier.campaignId, joinedModifier.adGroupId, true,
            enabledBannerTypeMultiplierAtoms + enabledInventoryMultiplierAtoms + defaultMultiplier)
    }

    private fun toMultiplierAtom(adjustment: BidModifierBannerTypeAdjustment): MultiplierAtom {
        val customExpression = CustomExpression.newBuilder()
            .setInventoryType(InventoryTypeEnum.MediaCreativeBanner)
            .build()
        return MultiplierAtom.newBuilder()
            .setMultiplier(adjustment.percent * MultiplierHandler.MULTIPLIER_COEF)
            .setCustomCondition(customExpression)
            .build()
    }

    private fun toMultiplierAtom(adjustment: BidModifierInventoryAdjustment): MultiplierAtom {
        val exportInventoryType = INVENTORY_TYPE_MAPPING[adjustment.inventoryType]

        // todo zakhar: тест на исчерпываемость конвертации
        checkNotNull(exportInventoryType, {
            "Unknown type ${adjustment.inventoryType} of BidModifierInventoryAdjustment with ID=${adjustment.id}"
        })
        val customExpression = CustomExpression.newBuilder()
            .setInventoryType(exportInventoryType)
            .build()
        return MultiplierAtom.newBuilder()
            .setMultiplier(adjustment.percent * MultiplierHandler.MULTIPLIER_COEF)
            .setCustomCondition(customExpression)
            .build()
    }

    private fun createDefaultMultiplierAtom(): MultiplierAtom {
        val customExpression = CustomExpression.newBuilder()
            .setInventoryType(InventoryTypeEnum.VideoDefault)
            .build()
        return MultiplierAtom.newBuilder()
            .setMultiplier(0)
            .setCustomCondition(customExpression)
            .build()
    }
}

private data class InventoryTypeKey(
    val campaignId: Long,
    val adGroupId: Long?
)

private data class JoinedInventoryTypeBidModifier(
    val key: InventoryTypeKey,
    val bannerType: BidModifierBannerType?,
    val inventory: BidModifierInventory?
) {
    val campaignId = key.campaignId
    val adGroupId = key.adGroupId

    /**
     * Хотя бы одна корректировка включена
     */
    val enabled: Boolean
        get() = (bannerType?.enabled ?: false) || (inventory?.enabled ?: false)
}
