package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.expression.MultiplierAtom;
import ru.yandex.adv.direct.expression.TargetingExpression;
import ru.yandex.adv.direct.expression.TargetingExpressionAtom;
import ru.yandex.adv.direct.expression.keywords.KeywordEnum;
import ru.yandex.adv.direct.expression.multipler.type.MultiplierTypeEnum;
import ru.yandex.adv.direct.expression.operations.OperationEnum;
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.DeleteInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierAndDeleteInfos;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class RetargetingMultiplierHandler implements BidModifierMultiplierHandler {
    public static final Integer RSYA_TARGET_TYPE = 3;

    @Override
    public MultiplierTypeEnum getExportMultiplierType() {
        return MultiplierTypeEnum.Retargeting;
    }

    @Override
    public MultiplierType getMultiplierType() {
        return MultiplierType.RETARGETING;
    }

    @Override
    public MultiplierAndDeleteInfos handle(int shard, Collection<? extends BidModifier> bidModifiers) {
        List<MultiplierInfo> multiplierInfos = new ArrayList<>();
        List<DeleteInfo> deleteInfos = new ArrayList<>();
        Map<RetargetingTypeMultiplierKey, ? extends List<? extends BidModifier>> groups = StreamEx.of(bidModifiers)
                .mapToEntry(m -> new RetargetingTypeMultiplierKey(m.getCampaignId(), m.getAdGroupId()), Function.identity())
                .grouping();

        EntryStream.of(groups)
                        .forKeyValue((key, modifiers) -> {
                            // есть некорректные записи в БД, без значения в retargeting_multiplier_values,
                            // в таких случаях getRetargetingAdjustments возвращает null
                            // будем пропускать такие записи
                            var retargetingModifiers = StreamEx.of(modifiers)
                                    .select(AbstractBidModifierRetargeting.class)
                                    .filter(RetargetingMultiplierHandler::modifierExistsAndEnabled)
                                            .collect(Collectors.toList());
                            if (retargetingModifiers.size() > 0) {
                                multiplierInfos.add(toMultiplierInfo(retargetingModifiers, key));
                            } else {
                                deleteInfos.add(new DeleteInfo(MultiplierType.RETARGETING, key.campaignId,
                                        key.adGroupId));
                            }
                        });
        return new MultiplierAndDeleteInfos(multiplierInfos, deleteInfos);
    }

    private MultiplierInfo toMultiplierInfo(List<AbstractBidModifierRetargeting> modifiers, RetargetingTypeMultiplierKey key) {
        Optional<BidModifierRetargeting> retargetingModifier = StreamEx.of(modifiers)
                .select(BidModifierRetargeting.class)
                .findFirst();
        Optional<BidModifierRetargetingFilter> retargetingFilterModifier = StreamEx.of(modifiers)
                .select(BidModifierRetargetingFilter.class)
                .findFirst();
        var multiplierAtomsLists =  StreamEx
                .of(retargetingModifier.map(BidModifierRetargeting::getRetargetingAdjustments)
                        .map(list -> filterList(list, AbstractBidModifierRetargetingAdjustment::getAccessible))
                        .map(this::toMultiplierAtomListForRetargeting))
                .append(StreamEx.of(retargetingFilterModifier
                        .map(BidModifierRetargetingFilter::getRetargetingAdjustments)
                        .map(list -> filterList(list, AbstractBidModifierRetargetingAdjustment::getAccessible))
                        .map(this::toMultiplierAtomListForRetargetingFilter))
                );
        List<MultiplierAtom> multiplierAtoms = StreamEx.of(multiplierAtomsLists)
                .flatMap(Collection::stream)
                .toList();

        return new MultiplierInfo(MultiplierType.RETARGETING, key.campaignId, key.adGroupId, Boolean.TRUE, multiplierAtoms);
    }

    private List<MultiplierAtom> toMultiplierAtomListForRetargeting(
            List<AbstractBidModifierRetargetingAdjustment> adjustments) {
        return mapList(adjustments, this::toMultiplierAtomForRetargeting);
    }

    private MultiplierAtom toMultiplierAtomForRetargeting(AbstractBidModifierRetargetingAdjustment adjustment) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.GoalContextId)
                                        .setOperation(OperationEnum.MatchGoalContext)
                                        .setValue(adjustment.getRetargetingConditionId().toString())
                                        .build()
                        )
        );
        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(targetingExpressionBuilder.build())
                .build();
    }

    private List<MultiplierAtom> toMultiplierAtomListForRetargetingFilter(
            List<AbstractBidModifierRetargetingAdjustment> adjustments) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        mapList(adjustments, adjustment -> targetingExpressionBuilder.addAND(
                        TargetingExpression.Disjunction.newBuilder()
                                .addOR(
                                        TargetingExpressionAtom.newBuilder()
                                                .setKeyword(KeywordEnum.GoalContextId)
                                                .setOperation(OperationEnum.NotMatchGoalContext)
                                                .setValue(adjustment.getRetargetingConditionId().toString())
                                                .build()
                                )
                )
        );

        targetingExpressionBuilder.addAND(
                TargetingExpression.Disjunction.newBuilder()
                        .addOR(
                                TargetingExpressionAtom.newBuilder()
                                        .setKeyword(KeywordEnum.PageTargetType)
                                        .setOperation(OperationEnum.NotEqual)
                                        .setValue(RSYA_TARGET_TYPE.toString())
                                        .build()
                        )
        );
        return Collections.singletonList(
                MultiplierAtom.newBuilder()
                        .setMultiplier(0)
                        .setCondition(targetingExpressionBuilder.build())
                        .build()
        );
    }

    //  проверка, что запись валидна и корректировка доступна
    private static boolean modifierExistsAndEnabled(AbstractBidModifierRetargeting modifier) {
        return modifier.getEnabled() &&
                modifier.getRetargetingAdjustments() != null && modifier.getRetargetingAdjustments().stream()
                .anyMatch(AbstractBidModifierRetargetingAdjustment::getAccessible);
    }

    private static class RetargetingTypeMultiplierKey {
        private final long campaignId;
        private final Long adGroupId;

        public RetargetingTypeMultiplierKey(long campaignId, Long adGroupId) {
            this.campaignId = campaignId;
            this.adGroupId = adGroupId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            RetargetingMultiplierHandler.RetargetingTypeMultiplierKey that = (RetargetingMultiplierHandler.RetargetingTypeMultiplierKey) o;
            return campaignId == that.campaignId &&
                    Objects.equals(adGroupId, that.adGroupId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(campaignId, adGroupId);
        }
    }
}
