package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.expression.MultiplierAtom;
import ru.yandex.adv.direct.expression.TargetingExpression;
import ru.yandex.adv.direct.expression.TargetingExpressionAtom;
import ru.yandex.adv.direct.expression.keywords.KeywordEnum;
import ru.yandex.adv.direct.expression.multipler.type.MultiplierTypeEnum;
import ru.yandex.adv.direct.expression.operations.OperationEnum;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.DeleteInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.libs.timetarget.HoursCoef;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.libs.timetarget.WeekdayType;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.CampaignWithTimeTarget;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierAndDeleteInfos;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo;

// https://wiki.yandex-team.ru/bannernajakrutilka/server/timetargeting/
@Component
public class TimeMultiplierHandler implements MultiplierHandler<CampaignWithTimeTarget> {
    @Override
    public MultiplierType getMultiplierType() {
        return MultiplierType.TIME_TARGET;
    }

    @Override
    public MultiplierTypeEnum getExportMultiplierType() {
        return MultiplierTypeEnum.Time;
    }

    @Override
    public MultiplierAndDeleteInfos handle(int shard, Collection<? extends CampaignWithTimeTarget> objects) {
        List<MultiplierInfo> multiplierInfos = new ArrayList<>();
        List<DeleteInfo> deleteInfos = new ArrayList<>();

        StreamEx.of(objects)
                .mapToEntry(CampaignWithTimeTarget::getCampaignId, CampaignWithTimeTarget::getTimeTarget)
                .forKeyValue((campaignId, timeTarget) -> {
                    List<MultiplierAtom> multipliers = createTimeTargetMultipliers(timeTarget);
                    if (multipliers.isEmpty()) {
                        // такая кампания не найдена или timeTarget на ней не задан
                        deleteInfos.add(new DeleteInfo(MultiplierType.TIME_TARGET, campaignId, null));
                    } else {
                        multiplierInfos.add(
                                new MultiplierInfo(
                                        MultiplierType.TIME_TARGET, campaignId, null, true, multipliers));
                    }
                });
        return new MultiplierAndDeleteInfos(multiplierInfos, deleteInfos);
    }

    // TODO Перейти на общий метод TimeTargetBsExportUtils.addTargetTimeCondition
    private List<MultiplierAtom> createTimeTargetMultipliers(@Nullable TimeTarget timeTarget) {
        if (timeTarget == null) {
            return List.of();
        }

        // коэффициенты по дням недели и набору часов
        Map<Integer, Map<WeekdayType, BitSet>> coefs = new HashMap<>();
        Map<WeekdayType, HoursCoef> weekdayCoefs = timeTarget.getWeekdayCoefs();

        boolean hasHolidayCoefs = weekdayCoefs.containsKey(WeekdayType.HOLIDAY);
        boolean hasWorkingWeekendCoefs = weekdayCoefs.containsKey(WeekdayType.WORKING_WEEKEND);

        // если заданы коэфициенты для праздника, нужно выгрузить и их
        int maxWeekday = hasHolidayCoefs ? 8 : 7;
        for (int weekday = 1; weekday <= maxWeekday; weekday++) {
            WeekdayType weekdayType = WeekdayType.getById(weekday);
            HoursCoef hoursCoef = weekdayCoefs.get(weekdayType);
            if (hoursCoef != null) {
                for (int hour = 0; hour < 24; hour++) {
                    Integer coef = hoursCoef.getCoefForHour(hour);
                    Map<WeekdayType, BitSet> weekdayTimetable = coefs.computeIfAbsent(coef, c -> new LinkedHashMap<>());
                    BitSet hours = weekdayTimetable.computeIfAbsent(weekdayType, d -> new BitSet(24));
                    hours.set(hour);
                }
            } else {
                Map<WeekdayType, BitSet> weekdayTimetable =
                        coefs.computeIfAbsent(TimeTarget.PredefinedCoefs.ZERO.getValue(), c -> new LinkedHashMap<>());
                BitSet hours = weekdayTimetable.computeIfAbsent(weekdayType, d -> new BitSet(24));
                for (int hour = 0; hour < 24; hour++) {
                    hours.set(hour);
                }
            }
        }

        if (Sets.difference(coefs.keySet(), Set.of(0, 100)).isEmpty()) {
            // Сохраняем поведение старого транспорта, где если в timeTarget только включение и отключение,
            // то коэффициенты для корректировки не присылаются
            return List.of();
        }

        return EntryStream.of(coefs)
                // сортируем по коэффициентам, т.к. хотим детерминированный порядок для тестов
                .sortedByInt(Map.Entry::getKey)
                .mapValues(weekdayTimetable -> {
                    // для каждого коэффициента, группируем дни недели с одинаковым
                    // расписанием включения этого коэффициента
                    return EntryStream.of(weekdayTimetable)
                            .invert()
                            .grouping();
                })
                .mapKeyValue(
                        (coef, hoursToWeekdayGroup) -> toMultiplierAtomTimeTargetCoef(
                                coef, hoursToWeekdayGroup, hasHolidayCoefs, hasWorkingWeekendCoefs))
                .toList();
    }

    private MultiplierAtom toMultiplierAtomTimeTargetCoef(
            Integer coef, Map<BitSet, List<WeekdayType>> hoursToWeekdayGroup,
            boolean hasHolidayCoefs, boolean hasWorkingWeekendCoefs) {
        TargetingExpression.Disjunction.Builder disjunctionBuilder = TargetingExpression.Disjunction.newBuilder();

        EntryStream.of(hoursToWeekdayGroup)
                .mapKeyValue(this::createTimePatternValue)
                .sorted() // хотим детерминированный порядок
                .forEach(timePatternValue -> disjunctionBuilder.addOR(
                        TargetingExpressionAtom.newBuilder()
                                // TimetableSimple учитывает переносы выходных -- рабочие выходные: если
                                // выходной, но
                                // по производственному календарю в него работают, как за среду, например, то
                                // будет
                                // браться коэфициенты со среды
                                // Timetable не учитывает рабочие выходные
                                .setKeyword(hasWorkingWeekendCoefs ?
                                        KeywordEnum.TimetableSimple : KeywordEnum.Timetable)
                                .setOperation(hasHolidayCoefs ?
                                        OperationEnum.TimeLike : OperationEnum.TimeIgnoreHolidayLike)
                                .setValue(timePatternValue)
                                .build()
                        )
                );
        return MultiplierAtom.newBuilder()
                .setMultiplier(coef * MULTIPLIER_COEF)
                .setCondition(TargetingExpression.newBuilder().addAND(disjunctionBuilder).build())
                .build();
    }

    @NotNull
    private String createTimePatternValue(BitSet hours, List<WeekdayType> weekdays) {
        // дни нумеруются с 1 (пн) по 7 (вс), 8 -- праздник, часы -- с A (0) до X (23).
        String hoursString = hours.stream()
                .map(hour -> (char) ('A' + hour))
                .collect(StringBuilder::new, (stringBuilder, i) -> stringBuilder.append((char) i),
                        StringBuilder::append)
                .toString();
        String weekdaysStr = weekdays.stream()
                .map(WeekdayType::getInternalNum)
                .map(Object::toString)
                .collect(Collectors.joining());
        return weekdaysStr + hoursString;
    }

}
