package ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.handler;

import java.util.Collection;
import java.util.List;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.expression.MultiplierAtom;
import ru.yandex.adv.direct.expression.TargetingExpression;
import ru.yandex.adv.direct.expression.TargetingExpressionAtom;
import ru.yandex.adv.direct.expression.keywords.KeywordEnum;
import ru.yandex.adv.direct.expression.multipler.type.MultiplierTypeEnum;
import ru.yandex.adv.direct.expression.operations.OperationEnum;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherLiteral;
import ru.yandex.direct.core.entity.bidmodifier.OperationType;
import ru.yandex.direct.core.entity.bidmodifier.WeatherType;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.container.MultiplierInfo;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class WeatherMultiplierHandler extends BaseMultiplierHandler {
    @Override
    public MultiplierType getMultiplierType() {
        return MultiplierType.WEATHER;
    }

    @Override
    public MultiplierTypeEnum getExportMultiplierType() {
        return MultiplierTypeEnum.Weather;
    }

    @Override
    public List<MultiplierInfo> convert(Collection<? extends BidModifier> bidModifiers) {
        return StreamEx.of(bidModifiers)
                .select(BidModifierWeather.class)
                .map(this::toMultiplierInfo)
                .toList();
    }

    private MultiplierInfo toMultiplierInfo(BidModifierWeather modifier) {
        return new MultiplierInfo(
                MultiplierType.WEATHER, modifier.getCampaignId(), modifier.getAdGroupId(), modifier.getEnabled(),
                mapList(modifier.getWeatherAdjustments(), this::toMultiplierAtom)
        );
    }

    private MultiplierAtom toMultiplierAtom(BidModifierWeatherAdjustment adjustment) {
        return MultiplierAtom.newBuilder()
                .setMultiplier(adjustment.getPercent() * MULTIPLIER_COEF)
                .setCondition(toCondition(adjustment.getExpression()))
                .build();
    }

    private TargetingExpression toCondition(List<List<BidModifierWeatherLiteral>> expression) {
        TargetingExpression.Builder targetingExpressionBuilder = TargetingExpression.newBuilder();
        for (List<BidModifierWeatherLiteral> disjunction : expression) {
            TargetingExpression.Disjunction.Builder disjunctionBuilder = TargetingExpression.Disjunction.newBuilder();
            for (BidModifierWeatherLiteral bidModifierWeatherLiteral : disjunction) {
                disjunctionBuilder.addOR(toTargetingExpressionAtom(bidModifierWeatherLiteral));
            }
            targetingExpressionBuilder.addAND(disjunctionBuilder.build());
        }
        return targetingExpressionBuilder.build();
    }

    private TargetingExpressionAtom toTargetingExpressionAtom(BidModifierWeatherLiteral bidModifierWeatherLiteral) {
        return TargetingExpressionAtom.newBuilder()
                .setKeyword(toKeyword(bidModifierWeatherLiteral.getParameter()))
                .setOperation(toOperation(bidModifierWeatherLiteral.getOperation()))
                .setValue(bidModifierWeatherLiteral.getValue().toString())
                .build();
    }

    private KeywordEnum toKeyword(WeatherType weatherType) {
        switch (weatherType) {
            case TEMP:
                return KeywordEnum.Temperature;
            case PREC_STRENGTH:
                return KeywordEnum.PrecipitationStrength;
            case CLOUDNESS:
                return KeywordEnum.Cloudness;
            default:
                throw new IllegalArgumentException(String.format("Unknown weather type %s", weatherType));
        }
    }

    private OperationEnum toOperation(OperationType operationType) {
        switch (operationType) {
            case GE:
                return OperationEnum.GreaterOrEqual;
            case LE:
                return OperationEnum.LessOrEqual;
            case EQ:
                return OperationEnum.Equal;
            default:
                throw new IllegalArgumentException(String.format("Unknown operation type %s", operationType));
        }
    }
}
