package ru.yandex.direct.logicprocessor.processors.bsexport.resources;

import java.time.Duration;
import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.bsexport.resources.BsExportBannerResourcesConfig;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.db.PpcPropertyNames.BS_EXPORT_ASSETS_ON_IMAGE_ADDITION_BY_BIDS;
import static ru.yandex.direct.common.db.PpcPropertyNames.BS_EXPORT_ASSETS_ON_IMAGE_ADDITION_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_BS_EXPORT;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_0, DIRECT_BS_EXPORT},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_DIRECT_BS_TRANSPORT_YT_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT},
                method = NotificationMethod.TELEGRAM))
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags =
        {DIRECT_PRIORITY_0})
@EssLogicProcessor(BsExportBannerResourcesConfig.class)
public class BsExportBannerResourcesProcessor extends BaseLogicProcessor<BsExportBannerResourcesObject> {

    private final BsExportBannerResourcesService bsExportBannerResourcesService;

    private final PpcProperty<Boolean> exportAssetsOnImageAdditionEnabledProp;
    private final PpcProperty<Set<Long>> bidsToExportAssetsOnImageAdditionProp;

    @Autowired
    public BsExportBannerResourcesProcessor(EssLogicProcessorContext essLogicProcessorContext,
                                            BsExportBannerResourcesService bsExportBannerResourcesService,
                                            PpcPropertiesSupport ppcPropertiesSupport) {
        super(essLogicProcessorContext);
        this.bsExportBannerResourcesService = bsExportBannerResourcesService;

        this.exportAssetsOnImageAdditionEnabledProp =
                ppcPropertiesSupport.get(BS_EXPORT_ASSETS_ON_IMAGE_ADDITION_ENABLED, Duration.ofMinutes(1));
        this.bidsToExportAssetsOnImageAdditionProp =
                ppcPropertiesSupport.get(BS_EXPORT_ASSETS_ON_IMAGE_ADDITION_BY_BIDS, Duration.ofMinutes(1));
    }

    @Override
    public void process(List<BsExportBannerResourcesObject> logicObjects) {
        bsExportBannerResourcesService.updateResources(getShard(), filterByImageAdditionProperties(logicObjects));
    }

    private List<BsExportBannerResourcesObject> filterByImageAdditionProperties(
            List<BsExportBannerResourcesObject> logicObjects) {
        if (!exportAssetsOnImageAdditionEnabledProp.getOrDefault(false)) {
            var bidsToExportAssetsOnImageAddition = bidsToExportAssetsOnImageAdditionProp.getOrDefault(Set.of());
            return logicObjects.stream()
                    .filter(o -> !o.isImageBannerExclusive()
                            || bidsToExportAssetsOnImageAddition.contains(o.getBid()))
                    .collect(toList());
        }
        return logicObjects;
    }
}
