package ru.yandex.direct.logicprocessor.processors.bsexport.resources.handler

import java.time.Clock
import java.time.Duration
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.adv.direct.banner.resources.BannerResources
import ru.yandex.direct.bstransport.yt.repository.resources.BaseBannerResourcesYtRepository
import ru.yandex.direct.bstransport.yt.utils.CaesarIterIdGenerator
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository
import ru.yandex.direct.core.entity.bs.common.service.BsBannerIdCalculator.calculateBsBannerId
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.BannerResource
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.BannerResourcesStat
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.IBannerResourceLoader

data class BannerResourcesExtended(
    val bannerResources: BannerResources,
    val cid: Long
)

abstract class BaseBannerResourceHandler<T> @Autowired constructor(
    private val bannerResourcesLoader: IBannerResourceLoader<T>,
    private val bannerResourcesYtRepository: BaseBannerResourcesYtRepository,
    private val logBsExportEssService: LogBsExportEssService,
    private val caesarIterIdGenerator: CaesarIterIdGenerator,
    private val bannerImageRepository: BannerImageRepository,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val clock: Clock = Clock.systemUTC()
) : IBannerResourcesHandler {
    companion object {
        private val logger = LoggerFactory.getLogger(BaseBannerResourceHandler::class.java)
        private const val BS_EXPORT_TYPE = "banner_resources"
    }

    private val exportAssetsForImageCopyBannerEnabledProp: PpcProperty<Boolean> = ppcPropertiesSupport
        .get(PpcPropertyNames.BS_EXPORT_ASSETS_FOR_IMAGE_COPY_BANNER_ENABLED, Duration.ofMinutes(1))
    private val bidsToExportAssetsForImageCopyBannerProp: PpcProperty<Set<Long>> = ppcPropertiesSupport
        .get(PpcPropertyNames.BS_EXPORT_ASSETS_FOR_IMAGE_COPY_BANNER_BY_BIDS, Duration.ofMinutes(1))

    private val bannerResourcesMonitoring = BannerResourcesMonitoring(bannerResourceType())
    protected abstract fun mapResourceToProto(): (T, BannerResources.Builder) -> Unit

    override fun handle(shard: Int, objects: Collection<BsExportBannerResourcesObject>) {
        try {
            handleInternal(shard, objects)
        } catch (e: RuntimeException) {
            logger.error("Failed to handle objects for type: ${bannerResourceType()}; " +
                "objects: ${if (objects.size > 1_000) objects.take(1_000) else objects}; " +
                "error: ${e.toString().take(100_000)}")
            throw e
        }
    }

    private fun handleInternal(shard: Int, objects: Collection<BsExportBannerResourcesObject>) {
        if (objects.isEmpty()) return

        val iterId = caesarIterIdGenerator.generateCaesarIterId()
        val updateTime = clock.instant().epochSecond

        val loaderResult = bannerResourcesLoader.loadResources(shard, objects)
        val bannerResources = loaderResult.resources
        if (bannerResources.isEmpty()) return

        // дублирование ресурсов для картиночных копий баннеров
        val resultingResources = duplicateResourcesForImageBanners(shard, bannerResources)
        loaderResult.stat.sent = resultingResources.size

        val extendedBannerResourcesList = resultingResources
            .map { bannerResource: BannerResource<T> ->
                val resourcesBuilder = BannerResources.newBuilder()
                    .setOrderId(bannerResource.orderId)
                    .setExportId(bannerResource.bid)
                    .setBannerId(bannerResource.bsBannerId)
                    .setAdgroupId(bannerResource.pid)
                    .setUpdateTime(updateTime)
                    .setIterId(iterId)
                mapResourceToProto()(bannerResource.resource, resourcesBuilder)

                BannerResourcesExtended(
                    bannerResources = resourcesBuilder.build(),
                    cid = bannerResource.cid
                )
            }

        if (extendedBannerResourcesList.isNotEmpty()) {
            val bannerResourcesList = extendedBannerResourcesList.map { it.bannerResources }
            bannerResourcesYtRepository.modify(bannerResourcesList)
        }
        sendStat(shard, loaderResult.stat)
        logResources(extendedBannerResourcesList)
    }

    /**
     * Дублирует баннерные ресурсы, полученные из loader-ов для родительского баннера, для картиночной копии баннера,
     * если она существует и включено проперти, отвечающее за отправку ресурсов на картиночные копии.
     *
     * Продублированные ресурсы содержат bid (ExportID) и bsBannerId (BannerID) картиночных копий.
     *
     * @return Возвращает список баннерных ресурсов для отправки с продублированными ресурсами для картиночных копий
     */
    private fun duplicateResourcesForImageBanners(
        shard: Int,
        bannerResources: MutableList<BannerResource<T>>
    ): List<BannerResource<T>> {
        // проперти для проверки экспорта ассетов на картиночных копиях баннеров
        val exportAssetsForImageCopyBannersEnabled = exportAssetsForImageCopyBannerEnabledProp.getOrDefault(false)
        val bidsToGetImageInfo = HashSet(bannerResources.map { it.bid })
        if (!exportAssetsForImageCopyBannersEnabled) {
            val bidsToExportAssetsForImageCopyBanner =
                bidsToExportAssetsForImageCopyBannerProp.getOrDefault(emptySet())
            bidsToGetImageInfo.retainAll(bidsToExportAssetsForImageCopyBanner)
        }
        val bidToBannerImagedBsData = bannerImageRepository.getBannerImageIdsFromBids(shard, bidsToGetImageInfo)

        val finalResources = mutableListOf<BannerResource<T>>()
        bannerResources.forEach { resource ->
            if (bidToBannerImagedBsData.containsKey(resource.bid)) {
                val bannerImageBsData = bidToBannerImagedBsData[resource.bid]!!
                val imageId = bannerImageBsData.imageId
                var imageBsBannerId = bannerImageBsData.bsImageBannerId

                imageBsBannerId = if (imageBsBannerId == 0L) {
                    if (!bannerImageBsData.singleAdToBs) {
                        calculateBsBannerId(imageId)
                    } else {
                        // В новой схеме картиночный баннер передаем в БК в родительском баннере
                        if (bannerImageBsData.bsBannerId > 0L) bannerImageBsData.bsBannerId
                        else calculateBsBannerId(bannerImageBsData.bid)
                    }
                } else imageBsBannerId
                val imageBannerResource = resource.getImageBannerResource(imageId, imageBsBannerId)

                finalResources.addAll(listOf(resource, imageBannerResource))
            } else {
                finalResources.add(resource)
            }
        }

        return finalResources
    }

    private fun sendStat(shard: Int, stat: BannerResourcesStat) {
        logger.info("${stat.candidates} candidates, ${stat.sent} sent objects for type ${bannerResourceType()}")
        bannerResourcesMonitoring.apply {
            addCandidatesToSendResources(shard, stat.candidates)
            addSentResources(shard, stat.sent)
        }
    }

    private fun logResources(extendedBannerResources: List<BannerResourcesExtended>) {
        val logsData = extendedBannerResources
            .map { extendedBannerResource: BannerResourcesExtended ->
                val bannerResources = extendedBannerResource.bannerResources
                LogBsExportEssData<BannerResources?>()
                    .withBid(bannerResources.exportId)
                    .withPid(bannerResources.adgroupId)
                    .withCid(extendedBannerResource.cid)
                    .withBsBannerId(bannerResources.bannerId)
                    .withOrderId(bannerResources.orderId)
                    .withData(bannerResources)
            }

        logBsExportEssService.logData(logsData, BS_EXPORT_TYPE)
    }
}
