package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.time.Clock;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.BannerResource;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.BannerResourcesStat;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bs.common.service.BsBannerIdCalculator.calculateBsBannerId;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
public class BannerDeleteLoader implements IBannerResourceLoader<Long> {
    private final BsOrderIdCalculator bsOrderIdCalculator;
    private final BannerTypedRepository bannerTypedRepository;
    private final Clock clock;

    public BannerDeleteLoader(BsOrderIdCalculator bsOrderIdCalculator, BannerTypedRepository bannerTypedRepository,
                              Clock clock) {
        this.bsOrderIdCalculator = bsOrderIdCalculator;
        this.bannerTypedRepository = bannerTypedRepository;
        this.clock = clock;
    }

    @Autowired
    public BannerDeleteLoader(BsOrderIdCalculator bsOrderIdCalculator, BannerTypedRepository bannerTypedRepository) {
        this(bsOrderIdCalculator, bannerTypedRepository, Clock.systemUTC());
    }

    @Override
    public LoaderResult<Long> loadResources(int shard, Collection<BsExportBannerResourcesObject> objects) {
        var bids = listToSet(objects, BsExportBannerResourcesObject::getBid);
        var existingBids = bannerTypedRepository.getSafely(shard, bids, Banner.class).stream()
                .map(Banner::getId)
                .collect(toSet());
        var notExistingObjects = objects.stream()
                .filter(o -> !existingBids.contains(o.getBid()))
                .collect(toList());

        var deletedTime = clock.instant().getEpochSecond();
        var cidToOrderIdMap = getCidsToOrderIdMap(shard, notExistingObjects);
        var resources = notExistingObjects.stream()
                .filter(object -> cidToOrderIdMap.containsKey(object.getCid()))
                .map(object ->
                        new BannerResource.Builder<Long>()
                                .setOrderId(cidToOrderIdMap.get(object.getCid()))
                                .setBsBannerId(getBannerId(object))
                                .setBid(object.getBid())
                                .setPid(object.getPid())
                                .setCid(object.getCid())
                                .setResource(deletedTime).build())
                .collect(Collectors.toList());
        var stat = new BannerResourcesStat()
                .setCandidates(objects.size())
                .setSent(resources.size());
        return new LoaderResult<>(resources, stat);

    }

    private long getBannerId(BsExportBannerResourcesObject object) {
        return object.getBannerId() == 0L ? calculateBsBannerId(object.getBid()) : object.getBannerId();
    }

    private Map<Long, Long> getCidsToOrderIdMap(int shard, Collection<BsExportBannerResourcesObject> objects) {
        var cids = objects.stream()
                .map(BsExportBannerResourcesObject::getCid)
                .collect(toList());
        return bsOrderIdCalculator.calculateOrderIdIfNotExist(shard, cids);
    }

}
