package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.banner.resources.FeedInfo
import ru.yandex.direct.core.bsexport.repository.resources.AdGroupFeedInfo
import ru.yandex.direct.core.bsexport.repository.resources.BsExportFeedInfoRepository
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.DynamicBanner
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.bs.common.service.BsBannerIdCalculator.calculateBsBannerId
import ru.yandex.direct.ess.common.utils.TablesEnum
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.BannerResource
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.getAdditionalInfoList
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.getLoaderResult

/**
 * Информация о фиде хранится на группе, но отправляем мы её как ресурс баннера.
 */
@Component
class BannerFeedInfoLoader(
    private val bannerResourcesHelper: BannerResourcesHelper,
    private val bsExportFeedInfoRepository: BsExportFeedInfoRepository,
    private val bannerTypedRepository: BannerTypedRepository
) : IBannerResourceLoader<FeedInfo> {

    private val logger = LoggerFactory.getLogger(BannerFeedInfoLoader::class.java)

    override fun loadResources(shard: Int, objects: Collection<BsExportBannerResourcesObject>): LoaderResult<FeedInfo> {
        val inputBannerIds = objects.mapNotNull { it.bid }
        val additionalBannerIds = obtainBannerIdsFromAdditionalInfo(shard, objects)
        val bannerIds = inputBannerIds + additionalBannerIds

        val banners = bannerTypedRepository.getSafely(shard, bannerIds, BannerWithSystemFields::class.java)

        val dynamicAdGroupIds = banners.asSequence()
            .filterIsInstance<DynamicBanner>()
            .mapNotNull { it.adGroupId }
            .toSet()

        val performanceAdGroupIds = banners.asSequence()
            .filterIsInstance<PerformanceBanner>()
            .mapNotNull { it.adGroupId }
            .toSet()

        // нет фильтрации по типу баннера т.к. в будущем планируем все типы баннеров уметь создавать в ТГО группе
        val textAdGroupIds = banners.asSequence()
            .mapNotNull { it.adGroupId }
            .toSet()

        // performance_main баннеры могут быть в performance и ТГО группах
        // выше для ТГО групп эти баннеры учитываются, а для performance групп надо отдельно добавить
        val performanceBannerMainAdGroupIds = banners.asSequence()
            .filterIsInstance<PerformanceBannerMain>()
            .mapNotNull { it.adGroupId }
            .toSet()

        val feedInfos = bsExportFeedInfoRepository.getFeedInfoForTextAdGroups(shard, textAdGroupIds) +
            bsExportFeedInfoRepository.getFeedInfoForDynamicFeedAdGroups(shard, dynamicAdGroupIds) +
            bsExportFeedInfoRepository.getFeedInfoForPerformanceAdGroups(
                shard,
                performanceAdGroupIds + performanceBannerMainAdGroupIds
            )

        val feedInfoByAdGroupId = feedInfos
            .map { it.adGroupId to convertFeedInfo(it) }
            .toMap()

        val cidToOrderId: Map<Long, Long> = bannerResourcesHelper.getCidsToOrderIdMap(shard, banners)

        val resources = banners.asSequence()
            .filter { it.campaignId in cidToOrderId }
            .map { createBannerResource(it, cidToOrderId[it.campaignId]!!, feedInfoByAdGroupId[it.adGroupId]) }
            .toList()

        return getLoaderResult(objects, resources)
    }

    private fun obtainBannerIdsFromAdditionalInfo(
        shard: Int,
        objects: Collection<BsExportBannerResourcesObject>
    ): List<Long> {
        val additionalInfoList = getAdditionalInfoList(objects)

        val adGroupIds = additionalInfoList
            .filter {
                (it.additionalTable == TablesEnum.ADGROUPS_TEXT) ||
                    (it.additionalTable == TablesEnum.ADGROUPS_DYNAMIC) ||
                    (it.additionalTable == TablesEnum.ADGROUPS_PERFORMANCE)
            }
            .map { it.additionalId }

        val feedIds = additionalInfoList
            .filter { it.additionalTable == TablesEnum.FEEDS }
            .map { it.additionalId }

        if (adGroupIds.size + feedIds.size != additionalInfoList.size) {
            logger.warn(
                "AdditionalInfo has unknown tables " +
                    "(additionalInfoList.size=${additionalInfoList.size}, adGroupIds.size=${adGroupIds.size})"
            )
        }

        val adGroupTextIdsFromFeeds = bsExportFeedInfoRepository.getAdGroupIdsByFeedIds(shard, feedIds)

        return bsExportFeedInfoRepository.getBannerIdsByAdGroupIds(
            shard, adGroupIds + adGroupTextIdsFromFeeds
        )
    }
}

private fun createBannerResource(
    banner: BannerWithSystemFields,
    orderId: Long,
    feedInfo: FeedInfo?
): BannerResource<FeedInfo> {
    val bsBannerId = if (banner.bsBannerId == 0L) calculateBsBannerId(banner.id) else banner.bsBannerId
    return BannerResource.Builder<FeedInfo>()
        .setBid(banner.id)
        .setPid(banner.adGroupId)
        .setCid(banner.campaignId)
        .setBsBannerId(bsBannerId)
        .setOrderId(orderId)
        .setResource(feedInfo)
        .build()
}

private fun convertFeedInfo(info: AdGroupFeedInfo) =
    FeedInfo.newBuilder()
        .setMarketBusinessId(info.marketBusinessId)
        .setMarketShopId(info.marketShopId)
        .setMarketFeedId(info.marketFeedId)
        .setDirectFeedId(info.directFeedId)
        .build()

