package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithHrefForBsExport;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.Language;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.HrefsInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.BsDomainIdGenerationService;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.DomainFilterService;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.HrefAndSite;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.HrefAndSiteService;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.MOBILE_CONTENT;
import static ru.yandex.direct.logicprocessor.processors.bsexport.platformname.BannerPlatformNameExtractor.getProtoBannerPlatformName;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
public class BannerHrefLoader extends BaseBannerResourcesLoader<BannerWithHrefForBsExport, HrefsInfo> {
    private static final Logger logger = LoggerFactory.getLogger(BannerHrefLoader.class);

    private final HrefAndSiteService hrefAndSiteService;
    private final HostingsHandler hostingsHandler;
    private final CampaignTypedRepository campaignTypedRepository;
    private final BsDomainIdGenerationService bsDomainIdGenerationService;
    private final DomainFilterService domainFilterService;
    private final AdGroupRepository adGroupRepository;

    public BannerHrefLoader(BannerResourcesLoaderContext context,
                            HostingsHandler hostingsHandler,
                            HrefAndSiteService hrefAndSiteService,
                            CampaignTypedRepository campaignTypedRepository,
                            BsDomainIdGenerationService bsDomainIdGenerationService,
                            DomainFilterService domainFilterService,
                            AdGroupRepository adGroupRepository) {
        super(context);
        this.hostingsHandler = hostingsHandler;
        this.hrefAndSiteService = hrefAndSiteService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.bsDomainIdGenerationService = bsDomainIdGenerationService;
        this.domainFilterService = domainFilterService;
        this.adGroupRepository = adGroupRepository;
    }

    @Override
    protected Class<BannerWithHrefForBsExport> getClassToLoadFromDb() {
        return BannerWithHrefForBsExport.class;
    }

    @Override
    protected Map<Long, HrefsInfo> getResources(int shard,
                                                List<BannerWithHrefForBsExport> bannersFromDb) {
        try {
            bannersFromDb = removeBannersWithMobileContentAdGroups(shard, bannersFromDb);
            if (bannersFromDb.isEmpty()) {
                return Map.of();
            }

            Map<Long, Language> bidToLanguage =
                    listToMap(bannersFromDb, Banner::getId, BannerWithSystemFields::getLanguage);

            var campaignIdToCampaigns = getCampaignIdToCampaignsMap(shard, bannersFromDb);

            var bannerIdsToHrefAndSite = StreamEx.of(bannersFromDb)
                    .filter(banner -> campaignIdToCampaigns.containsKey(banner.getCampaignId()))
                    .filter(this::hasReadyResource)
                    .toMap(Banner::getId,
                            banner -> hrefAndSiteService
                                    .extract(banner, campaignIdToCampaigns.get(banner.getCampaignId())));


            var sitesAscii = bannerIdsToHrefAndSite.values().stream()
                    .map(HrefAndSite::getSiteAscii)
                    .collect(toSet());

            var domainFilters = domainFilterService.getDomainsFilters(sitesAscii);
            var siteFilters = sitesAscii.stream()
                    .collect(toMap(siteAscii -> siteAscii, hostingsHandler::stripWww));

            var domainsToFilterIds = getFilterIds(domainFilters.values(), siteFilters.values());

            return bannerIdsToHrefAndSite.entrySet().stream()
                    .collect(toMap(Map.Entry::getKey, e -> {
                        Long bid = e.getKey();
                        HrefAndSite hrefAndSite = e.getValue();
                        var siteAscii = hrefAndSite.getSiteAscii();
                        var domainFilter = domainFilters.get(siteAscii);
                        var siteFilter = siteFilters.get(siteAscii);
                        long domainFilterId = domainsToFilterIds.getOrDefault(domainFilter, 0L);
                        long siteFilterId = domainsToFilterIds.getOrDefault(siteFilter, 0L);
                        return HrefsInfo.builder()
                                .withHref(hrefAndSite.getHref())
                                .withSite(hrefAndSite.getSite())
                                .withSiteFilter(siteFilter)
                                .withDomainFilter(domainFilter)
                                .withDomainFilterId(domainFilterId)
                                .withSiteFilterId(siteFilterId)
                                .withPlatformName(getProtoBannerPlatformName(hrefAndSite.getHref(),
                                        bidToLanguage.get(bid)))
                                .build();
                    }));
            // Для первого запуска, чтобы не останавливать транспорт целиком, попробуем завернуть загрузку в try-catch
            // уберем тут https://st.yandex-team.ru/DIRECT-130967
        } catch (RuntimeException e) {
            var bids = bannersFromDb.stream().map(BannerWithHrefForBsExport::getId).collect(toList());
            logger.error("Failed to handle hrefs for bids " + bids, e);
            return Map.of();
        }
    }

    private boolean hasReadyResource(BannerWithHrefForBsExport resourceFromDb) {
        return BannerStatusModerate.YES.equals(resourceFromDb.getStatusModerate());
    }

    private Map<String, Long> getFilterIds(Collection<String> domainFilters, Collection<String> siteFilters) {
        var filtersToGetIds = new HashSet<>(domainFilters);
        filtersToGetIds.addAll(siteFilters);
        return bsDomainIdGenerationService.generate(filtersToGetIds);
    }

    /**
     * Группы с типом mobile_content обрабатываются отдельно в {@link BannerMobileContentLoader}
     */
    private List<BannerWithHrefForBsExport> removeBannersWithMobileContentAdGroups(int shard,
                                                                                   List<BannerWithHrefForBsExport> banners) {
        var adGroupIds = listToSet(banners, BannerWithHrefForBsExport::getAdGroupId);
        var adGroupToTypeMap = adGroupRepository.getAdGroupTypesByIds(shard, adGroupIds);
        return StreamEx.of(banners)
                .filter(b -> adGroupToTypeMap.containsKey(b.getAdGroupId()))
                .filter(b -> !MOBILE_CONTENT.equals(adGroupToTypeMap.get(b.getAdGroupId())))
                .toList();
    }

    private Map<Long, CommonCampaign> getCampaignIdToCampaignsMap(int shard,
                                                                  Collection<BannerWithHrefForBsExport> banners) {
        var campaignIds = banners.stream()
                .map(BannerWithHrefForBsExport::getCampaignId)
                .collect(toList());

        return campaignTypedRepository.getSafely(shard, campaignIds, CommonCampaign.class).stream()
                .collect(toMap(CommonCampaign::getId, campaign -> campaign));
    }
}
