package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Component;

import ru.yandex.adv.direct.banner.resources.Format;
import ru.yandex.adv.direct.banner.resources.Logo;
import ru.yandex.direct.core.entity.banner.model.BannerWithLogoExtended;
import ru.yandex.direct.core.entity.banner.model.BannerWithLogoForBsExport;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.image.service.ImageUtils;
import ru.yandex.direct.validation.Predicates;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.model.BannerLogoStatusModerate.NO;
import static ru.yandex.direct.core.entity.banner.model.BannerLogoStatusModerate.YES;

@Component
public class BannerLogosLoader extends BaseBannerResourcesLoader<BannerWithLogoForBsExport, Logo> {
    private static final Set<String> FORMATS_TO_SENT = Set.of("x80");
    private final BannerImageFormatRepository bannerImageFormatRepository;

    public BannerLogosLoader(BannerResourcesLoaderContext context,
                             BannerImageFormatRepository bannerImageFormatRepository) {
        super(context);
        this.bannerImageFormatRepository = bannerImageFormatRepository;
    }

    @Override
    public Class<BannerWithLogoForBsExport> getClassToLoadFromDb() {
        return BannerWithLogoForBsExport.class;
    }

    @Override
    protected Map<Long, Logo> getResources(int shard, List<BannerWithLogoForBsExport> bannersFromDb) {
        Map<Long, Logo> logoMap = new HashMap<>();
        bannersFromDb.stream()
                .filter(banner -> NO.equals(banner.getLogoStatusModerate()))
                .forEach(banner -> logoMap.put(banner.getId(), null));

        var bannerWithLogoStatusModerateYes = bannersFromDb.stream()
                .filter(banner -> YES.equals(banner.getLogoStatusModerate()))
                .collect(toList());

        logoMap.putAll(getResourcesForModeratedLogos(shard, bannerWithLogoStatusModerateYes));
        return logoMap;
    }

    private Map<Long, Logo> getResourcesForModeratedLogos(int shard,
                                                          List<BannerWithLogoForBsExport> banners) {
        Map<Long, Logo> resultMap = new HashMap<>();
        var imagesHashes = banners.stream()
                .map(BannerWithLogoForBsExport::getLogoImageHash)
                .collect(toList());
        var hashToFormat = bannerImageFormatRepository.getBannerImageFormats(shard, imagesHashes);
        banners
                .forEach(banner -> {
                    boolean isLogoExtended = banner instanceof BannerWithLogoExtended;
                    var logo = Logo.newBuilder()
                            .addAllFormats(getFormats(hashToFormat.get(banner.getLogoImageHash()), isLogoExtended))
                            .build();
                    resultMap.put(banner.getId(), logo);
                });

        return resultMap;
    }

    private List<Format> getFormats(BannerImageFormat bannerImageFormat, boolean isLogoExtended) {
        checkArgument(Objects.nonNull(bannerImageFormat));
        return EntryStream.of(bannerImageFormat.getFormats())
                .filterKeys(isLogoExtended ? Predicates.ignore() : FORMATS_TO_SENT::contains)
                .mapKeyValue((formatId, format) -> Format.newBuilder()
                        .setFormat(formatId)
                        .setHeight(format.getHeight())
                        .setWidth(format.getWidth())
                        .setUrl(ImageUtils.generateImageSecureUrl(bannerImageFormat, formatId))
                        .build())
                .toList();
    }
}
