package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.bsexport.repository.resources.BsExportMobileContentRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithMobileContentDataForBsExport;
import ru.yandex.direct.core.entity.banner.model.BaseBannerWithResourcesForBsExport;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.HrefAndSiteService;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.mobilecontent.TrackerHrefHandleService;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
public class BannerMobileContentDataLoader extends BaseBannerResourcesLoader<BannerWithMobileContentDataForBsExport, String> {
    private static final Logger logger = LoggerFactory.getLogger(BannerMobileContentDataLoader.class);

    private final BsExportMobileContentRepository exportMobileContentRepository;
    private final TrackerHrefHandleService trackerHrefHandleService;
    private final HrefAndSiteService hrefAndSiteService;


    public BannerMobileContentDataLoader(BannerResourcesLoaderContext context,
                                         BsExportMobileContentRepository exportMobileContentRepository,
                                         TrackerHrefHandleService trackerHrefHandleService,
                                         HrefAndSiteService hrefAndSiteService) {
        super(context);
        this.exportMobileContentRepository = exportMobileContentRepository;
        this.trackerHrefHandleService = trackerHrefHandleService;
        this.hrefAndSiteService = hrefAndSiteService;
    }

    @Override
    protected Class<BannerWithMobileContentDataForBsExport> getClassToLoadFromDb() {
        return BannerWithMobileContentDataForBsExport.class;
    }

    @Override
    protected Map<Long, String> getResources(int shard,
                                             List<BannerWithMobileContentDataForBsExport> bannersFromDb) {
        try {
            var adGroupIds = listToSet(bannersFromDb, BannerWithMobileContentDataForBsExport::getAdGroupId);
            // мобильный контент на группе
            var mobileContentByAdgroupId = exportMobileContentRepository.getMobileContentByAdGroupIds(shard,
                    adGroupIds);

            var campaignIds = listToSet(bannersFromDb, BannerWithMobileContentDataForBsExport::getCampaignId);

            var campaigns = StreamEx.of(exportMobileContentRepository.getCampaigns(shard, campaignIds))
                    .mapToEntry(CommonCampaign::getId, c -> c)
                    .toMap();

            // берем только баннеры, у которых для группы есть запись в adgroups_mobile_content и у
            // которых есть запись в campaigns(за время подготовки кампания не была удалена)
            var bannersToCalculate = filterList(bannersFromDb,
                    banner -> mobileContentByAdgroupId.containsKey(banner.getAdGroupId())
                            && campaigns.containsKey(banner.getCampaignId()));

            if (bannersToCalculate.isEmpty()) {
                return Map.of();
            }

            return StreamEx.of(bannersToCalculate)
                    .mapToEntry(BannerWithMobileContentDataForBsExport::getId, b -> b)
                    .mapValues(b -> getImpressionUrl(b, campaigns.get(b.getCampaignId()), mobileContentByAdgroupId.get(b.getAdGroupId())))
                    .toMap();
            // Для первого запуска, чтобы не останавливать транспорт целиком, попробуем завернуть загрузку в try-catch
            // уберем тут https://st.yandex-team.ru/DIRECT-130967
        } catch (RuntimeException e) {
            var bids = bannersFromDb.stream().map(BaseBannerWithResourcesForBsExport::getId).collect(toList());
            logger.error("Failed to handle impression url for bids " + bids, e);
            return Map.of();
        }
    }

    /**
     * Получить трекерную ссылку на показ
     * Если на баннере указана ссылка то:
     * - к ней добавляются параметры в зависимости от трекинговой системы приложения
     * - происходит подстановка макросов БК и некоторых параметров
     * Если на баннере ссылки нет, то вернется null
     */
    @Nullable
    private String getImpressionUrl(BannerWithMobileContentDataForBsExport banner, CommonCampaign campaign,
                                    MobileContent mobileContent) {
        if (!hrefAndSiteService.isValidHref(banner.getImpressionUrl())) {
            return "";
        }
        var url = hrefAndSiteService.prepareHref(banner.getImpressionUrl());
        var urlWithTrackerParams = trackerHrefHandleService.handleHref(url, mobileContent.getOsType());
        var result = hrefAndSiteService.expandHref(urlWithTrackerParams, banner, campaign);
        return result != null ? result : "";
    }
}
