package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.net.IDN;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.bsexport.repository.resources.BsExportMobileContentRepository;
import ru.yandex.direct.core.bsexport.resources.model.MobileAppForBsExport;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithMobileContentAdGroupForBsExport;
import ru.yandex.direct.core.entity.banner.model.BaseBannerWithResourcesForBsExport;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.ess.common.utils.TablesEnum;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.AdditionalInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.MobileContentInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.MobileAppStoreHrefParser;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.DomainFilterService;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.HrefAndSiteService;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.mobilecontent.TrackerHrefHandleService;
import ru.yandex.direct.utils.model.UrlParts;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.ess.common.utils.TablesEnum.MOBILE_CONTENT;
import static ru.yandex.direct.logicprocessor.processors.bsexport.platformname.BannerPlatformNameExtractor.getProtoBannerPlatformName;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
public class BannerMobileContentLoader extends BaseBannerResourcesLoader<BannerWithMobileContentAdGroupForBsExport,
        MobileContentInfo> {
    private static final Logger logger = LoggerFactory.getLogger(BannerMobileContentLoader.class);

    private static final Set<TablesEnum> TABLES_WITH_CID = Set.of(TablesEnum.CAMPAIGNS_MOBILE_CONTENT);
    private final DomainFilterService domainFilterService;
    private final BsExportMobileContentRepository exportMobileContentRepository;
    private final TrackerHrefHandleService trackerHrefHandleService;
    private final HrefAndSiteService hrefAndSiteService;


    public BannerMobileContentLoader(BannerResourcesLoaderContext context,
                                     DomainFilterService domainFilterService,
                                     BsExportMobileContentRepository exportMobileContentRepository,
                                     TrackerHrefHandleService trackerHrefHandleService,
                                     HrefAndSiteService hrefAndSiteService) {
        super(context);
        this.domainFilterService = domainFilterService;
        this.exportMobileContentRepository = exportMobileContentRepository;
        this.trackerHrefHandleService = trackerHrefHandleService;
        this.hrefAndSiteService = hrefAndSiteService;
    }

    /**
     * Получение bids для записей из таблицы campaigns_mobile_content, adgroups_mobile_content и mobile_content
     */
    @Override
    List<Long> getAdditionalBids(int shard, Collection<AdditionalInfo> additionalInfoList) {
        var cids = getAdditionalIdsWithFilter(additionalInfoList,
                additionalInfo -> TABLES_WITH_CID.contains(additionalInfo.getAdditionalTable()));

        var mobileContentIds = getAdditionalIdsWithFilter(additionalInfoList,
                additionalInfo -> MOBILE_CONTENT.equals((additionalInfo.getAdditionalTable())));

        Set<Long> adGroupIds = new HashSet<>(
                exportMobileContentRepository.getAdgroupIdsForMobileContentIds(shard, mobileContentIds));
        Set<Long> bids = new HashSet<>();
        bids.addAll(exportMobileContentRepository.getBannerIdsByCampaignIds(shard, cids));
        bids.addAll(exportMobileContentRepository.getBannerIdsByAdGroupIds(shard, adGroupIds));
        return new ArrayList<>(bids);

    }

    private Set<Long> getAdditionalIdsWithFilter(Collection<AdditionalInfo> additionalInfoList,
                                                 Predicate<AdditionalInfo> predicate) {
        return additionalInfoList.stream()
                .filter(predicate)
                .map(AdditionalInfo::getAdditionalId)
                .collect(Collectors.toSet());
    }

    @Override
    protected Class<BannerWithMobileContentAdGroupForBsExport> getClassToLoadFromDb() {
        return BannerWithMobileContentAdGroupForBsExport.class;
    }

    @Override
    protected Map<Long, MobileContentInfo> getResources(int shard,
                                                        List<BannerWithMobileContentAdGroupForBsExport> bannersFromDb) {
        try {
            var adGroupIds = listToSet(bannersFromDb, BannerWithMobileContentAdGroupForBsExport::getAdGroupId);
            // мобильный контент на группе
            var mobileContentByAdgroupId = exportMobileContentRepository.getMobileContentByAdGroupIds(shard,
                    adGroupIds);

            var campaignIds = listToSet(bannersFromDb, BannerWithMobileContentAdGroupForBsExport::getCampaignId);

            var campaigns = StreamEx.of(exportMobileContentRepository.getCampaigns(shard, campaignIds))
                    .mapToEntry(CommonCampaign::getId, c -> c)
                    .toMap();

            // берем только промодерированные баннеры, у которых для группы есть запись в adgroups_mobile_content и у
            // которых есть запись в campaigns(за время подготовки кампания не была удалена)
            var bannersToCalculate = filterList(bannersFromDb,
                    banner -> hasReadyResource(banner)
                            && mobileContentByAdgroupId.containsKey(banner.getAdGroupId())
                            && campaigns.containsKey(banner.getCampaignId()));

            if (bannersToCalculate.isEmpty()) {
                return Map.of();
            }

            // информация о мобильном приложении на кампании
            var mobileAppByCampaignId = exportMobileContentRepository.getMobileAppsForCampaigns(shard, campaignIds);

            // ссылки из adgroups_mobile_content.store_content_href
            var adgroupIdToStoreContentHref = exportMobileContentRepository.getStoreUrlByAdGroupIds(shard, adGroupIds);

            var bidToHrefMap = StreamEx.of(bannersToCalculate)
                    .mapToEntry(BannerWithMobileContentAdGroupForBsExport::getId, b -> b)
                    .mapValues(b -> {
                        var href = getHref(b, campaigns.get(b.getCampaignId()),
                                mobileContentByAdgroupId.get(b.getAdGroupId()));
                        if (Objects.nonNull(href)) {
                            return href;
                        }
                        // если ссылки у баннера нет, то берется ссылка на мобильный контент в сторе
                        return adgroupIdToStoreContentHref.get(b.getAdGroupId());
                    })
                    .toMap();

            // siteFilter это mobile_content.store_content_id или mobile_content.bundle_id в зависимости от ОС
            var bidToSiteFilterMap = StreamEx.of(bannersToCalculate)
                    .mapToEntry(BannerWithMobileContentAdGroupForBsExport::getId, b -> b)
                    .mapValues(b -> getStoreAppId(mobileContentByAdgroupId.get(b.getAdGroupId())))
                    .mapValues(storeAppId ->
                            hrefAndSiteService.isValidDomain(storeAppId) ? IDN.toASCII(storeAppId) : "")
                    .toMap();

            var bidToPublisherDomainFilter =
                    getBidToPublisherDomainFilterMap(bannersToCalculate, mobileContentByAdgroupId,
                            mobileAppByCampaignId);

            // DomainFilter это либо publisher_domain, либо siteFilter
            var bidToDomainFilter = StreamEx.of(bannersToCalculate)
                    .mapToEntry(BannerWithMobileContentAdGroupForBsExport::getId, b -> b)
                    .mapValues(b -> bidToPublisherDomainFilter.containsKey(b.getId()) ?
                            // если publisher_domain некорректный, то его главное зеркало может быть пустым
                            bidToPublisherDomainFilter.get(b.getId()) : bidToSiteFilterMap.get(b.getId()))
                    .toMap();


            return StreamEx.of(bannersToCalculate)
                    .mapToEntry(BannerWithMobileContentAdGroupForBsExport::getId, b -> b)
                    .mapValues(b -> {
                        var mobileContent = mobileContentByAdgroupId.get(b.getAdGroupId());
                        var mobileApp = mobileAppByCampaignId.get(b.getCampaignId());
                        return MobileContentInfo.builder()
                                .withId(mobileContent.getId())
                                .withBundleId(getBsExportBundleId(mobileContent))
                                .withOsType(mobileContent.getOsType())
                                .withHref(bidToHrefMap.getOrDefault(b.getId(), ""))
                                // site это домен из adgroups_mobile_content.store_app_id
                                .withSite(getSite(adgroupIdToStoreContentHref.getOrDefault(b.getAdGroupId(), "")))
                                .withSiteFilter(bidToSiteFilterMap.getOrDefault(b.getId(), ""))
                                .withDomainFilter(bidToDomainFilter.getOrDefault(b.getId(), ""))
                                .withPlatformName(getProtoBannerPlatformName(
                                        adgroupIdToStoreContentHref.get(b.getAdGroupId()), b.getLanguage()))
                                .withLang(MobileAppStoreHrefParser.parseLang(mobileContent.getOsType(), mobileApp.getStoreHref()))
                                .withRegion(MobileAppStoreHrefParser.parseRegion(mobileContent.getOsType(), mobileApp.getStoreHref()))
                                .build();
                    })
                    .toMap();
            // Для первого запуска, чтобы не останавливать транспорт целиком, попробуем завернуть загрузку в try-catch
            // уберем тут https://st.yandex-team.ru/DIRECT-130967
        } catch (RuntimeException e) {
            var bids = bannersFromDb.stream().map(BaseBannerWithResourcesForBsExport::getId).collect(toList());
            logger.error("Failed to handle mobile content for bids " + bids, e);
            return Map.of();
        }
    }

    String getBsExportBundleId(MobileContent mobileContent) {
        return nvl(getStoreAppId(mobileContent), "");
    }

    String getStoreAppId(MobileContent mobileContent) {
        return OsType.IOS.equals(mobileContent.getOsType()) ? mobileContent.getBundleId() :
                mobileContent.getStoreContentId();
    }

    /**
     * Получить ссылку для мобильного контента
     * Если на баннере указана ссылка то:
     * - к ней добавляются параметры в зависимости от трекинговой системы приложения
     * - происходит подстановка макросов БК и некоторых параметров
     * Если на баннере ссылки нет, то вернется null
     */
    @Nullable
    private String getHref(BannerWithMobileContentAdGroupForBsExport banner, CommonCampaign campaign,
                           MobileContent mobileContent) {
        if (!hrefAndSiteService.isValidHref(banner.getHref())) {
            return null;
        }
        var href = hrefAndSiteService.prepareHref(banner.getHref());
        var hrefWithTrackerParams = trackerHrefHandleService.handleHref(href, mobileContent.getOsType());
        return hrefAndSiteService.extract(hrefWithTrackerParams, banner, campaign).getHref();
    }

    Map<Long, String> getBidToPublisherDomainFilterMap(
            List<BannerWithMobileContentAdGroupForBsExport> bannersFromDb,
            Map<Long, MobileContent> mobileContentByAdgroupId,
            Map<Long, MobileAppForBsExport> mobileAppByCampaignId) {

        var bidToPublisherDomainIdMap = StreamEx.of(bannersFromDb)
                .mapToEntry(
                        BannerWithMobileContentAdGroupForBsExport::getId,
                        b -> getPublisherDomainId(mobileContentByAdgroupId.get(b.getAdGroupId()),
                                mobileAppByCampaignId.get(b.getCampaignId())))
                .filterValues(Objects::nonNull)
                .toMap();


        var domainNameToDomainId =
                exportMobileContentRepository.getDomainsByIdsFromDict(Set.copyOf(bidToPublisherDomainIdMap.values()))
                        .stream()
                        .collect(toMap(Domain::getDomain, Domain::getId));

        var domainToDomainFilterMap = domainFilterService.getDomainsFilters(domainNameToDomainId.keySet());

        var publisherDomainIdToDomainFilterMap = EntryStream.of(domainToDomainFilterMap)
                .mapKeys(domainNameToDomainId::get)
                .toMap();

        return EntryStream.of(bidToPublisherDomainIdMap)
                .filterValues(publisherDomainIdToDomainFilterMap::containsKey)
                .mapValues(publisherDomainIdToDomainFilterMap::get)
                .toMap();

    }

    Long getPublisherDomainId(MobileContent mobileContent, @Nullable MobileAppForBsExport mobileApp) {
        if (Objects.nonNull(mobileApp)
                && mobileContent.getId().equals(mobileApp.getMobileContentId())
                && Objects.nonNull(mobileApp.getDomainId())) {
            return mobileApp.getDomainId();
        }
        return mobileContent.getPublisherDomainId();
    }

    private String getSite(String href) {
        if (!hrefAndSiteService.isValidHref(href)) {
            return "";
        }
        return UrlParts.fromUrl(href).getDomain();
    }

    private boolean hasReadyResource(BaseBannerWithResourcesForBsExport resourceFromDb) {
        return BannerStatusModerate.YES.equals(resourceFromDb.getStatusModerate());
    }
}
