package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.banner.resources.Multicard
import ru.yandex.adv.direct.banner.resources.MulticardSet
import ru.yandex.direct.core.entity.banner.model.BannerMulticard
import ru.yandex.direct.core.entity.banner.model.BannerMulticardSetStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSetForBsExport
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository
import ru.yandex.direct.core.entity.banner.type.multicard.BannerMulticardSetsRepository
import ru.yandex.direct.core.entity.image.model.Image
import ru.yandex.direct.core.entity.image.model.ImagePoolKey
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.ess.common.utils.TablesEnum
import ru.yandex.direct.ess.logicobjects.bsexport.resources.AdditionalInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.BannerImageConverter

@Component
class BannerMulticardSetLoader(
    bannerResourcesLoaderContext: BannerResourcesLoaderContext,
    private val bannerMulticardSetsRepository: BannerMulticardSetsRepository,
    private val bannerRelationsRepository: BannerRelationsRepository,
    private val imageDataRepository: ImageDataRepository,
) : BaseBannerResourcesLoader<BannerWithMulticardSetForBsExport, MulticardSet?>(
    bannerResourcesLoaderContext,
) {
    override fun getClassToLoadFromDb() = BannerWithMulticardSetForBsExport::class.java

    override fun getAdditionalBids(shard: Int, objects: Collection<AdditionalInfo>): List<Long> {
        val multicardIds = objects
            .filter { it.additionalTable == TablesEnum.BANNER_MULTICARDS }
            .map { it.additionalId }
            .toSet()

        return bannerMulticardSetsRepository.getBannerIdsByMulticardIds(shard, multicardIds).toList()
    }

    override fun getResources(
        shard: Int,
        bannerWithResourceFromDb: List<BannerWithMulticardSetForBsExport>
    ): Map<Long, MulticardSet?> {
        val result: MutableMap<Long, MulticardSet?> = mutableMapOf()

        val rejectedMulticardSets = bannerWithResourceFromDb
            .filter { it.multicardSetStatusModerate == BannerMulticardSetStatusModerate.NO }

        val acceptedMulticardSets = bannerWithResourceFromDb
            .filter { it.multicardSetStatusModerate == BannerMulticardSetStatusModerate.YES }

        val clientIdByBid = bannerRelationsRepository
            .getClientIdsByBannerIds(shard, acceptedMulticardSets.map { it.id })

        val imageKeys: List<ImagePoolKey> = acceptedMulticardSets
            .flatMap { banner ->
                banner.multicards.map { multicard ->
                    ImagePoolKey(clientIdByBid[banner.id]!!, multicard.imageHash)
                }
            }

        val images: Map<ImagePoolKey, Image> =
            imageDataRepository.getImages(shard, imageKeys)

        rejectedMulticardSets
            .forEach { result[it.id] = null }

        acceptedMulticardSets
            .forEach { result[it.id] = toMulticardSet(clientIdByBid[it.id]!!, it, images) }

        return result
    }

    private fun toMulticardSet(
        clientId: ClientId,
        banner: BannerWithMulticardSetForBsExport,
        images: Map<ImagePoolKey, Image>,
    ): MulticardSet {
        val cards: List<Multicard> = banner.multicards.mapIndexed { order, multicard ->
            val imageKey = ImagePoolKey(clientId, multicard.imageHash)
            toMulticard(order, multicard, images[imageKey])
        }

        return MulticardSet.newBuilder().addAllCards(cards).build()
    }

    private fun toMulticard(
        order: Int,
        multicard: BannerMulticard,
        image: Image?,
    ): Multicard {
        checkNotNull(image) {
            "No banner image format for multicard ${multicard.multicardId} and imageHash ${multicard.imageHash}"
        }

        val images = BannerImageConverter.toImages(image)
        val parsedMdsMeta = BannerImageConverter.toMdsMeta(image)

        return Multicard.newBuilder()
            .setOrder(order)
            .apply { if (multicard.text != null) text = multicard.text }
            .apply { if (multicard.href != null) href = multicard.href }
            .apply { if (multicard.price != null) price = multicard.price }
            .apply { if (multicard.priceOld != null) priceOld = multicard.priceOld }
            .apply { if (multicard.currency != null) currency = multicard.currency.name }
            .addAllImage(images)
            .apply { if (parsedMdsMeta != null) mdsMeta = parsedMdsMeta }
            .build()
    }

}
