package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.bsexport.repository.resources.BsExportBannerPermalinksRepository;
import ru.yandex.direct.core.bsexport.resources.model.BannerPermalink;
import ru.yandex.direct.core.bsexport.resources.model.PermalinkAssignType;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BaseBannerWithResourcesForBsExport;
import ru.yandex.direct.core.entity.banner.model.CpmGeoPinBanner;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.AdditionalInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.PermalinksInfo;

import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.bsexport.resources.model.PermalinkAssignType.AUTO;
import static ru.yandex.direct.core.bsexport.resources.model.PermalinkAssignType.MANUAL;
import static ru.yandex.direct.core.bsexport.resources.model.StatusPublish.PUBLISHED;

@Component
public class BannerPermalinksLoader extends BaseBannerResourcesLoader<BaseBannerWithResourcesForBsExport,
        PermalinksInfo> {
    private static final String HREF_TEMPLATE_FOR_CPM_GEO_PIN = "https://yandex.ru/profile/%d";
    private static final String SITE_FOR_CPM_GEO_PIN = "yandex.ru";
    private static final String DOMAIN_FILTER_TEMPLATE_FOR_CPM_GEO_PIN = "%d.ya-profile";

    private static final int PERMALINKS_SELECT_CHUNK_SIZE = SqlUtils.TYPICAL_SELECT_CHUNK_SIZE;

    private final BsExportBannerPermalinksRepository bannerPermalinksRepository;

    public BannerPermalinksLoader(BannerResourcesLoaderContext context,
                                  BsExportBannerPermalinksRepository bannerPermalinksRepository) {
        super(context);
        this.bannerPermalinksRepository = bannerPermalinksRepository;
    }

    @Override
    protected Class<BaseBannerWithResourcesForBsExport> getClassToLoadFromDb() {
        return BaseBannerWithResourcesForBsExport.class;
    }


    @Override
    List<Long> getAdditionalBids(int shard, Collection<AdditionalInfo> additionalInfoList) {
        var permalinks = additionalInfoList.stream()
                .map(AdditionalInfo::getAdditionalId)
                .collect(toSet());
        return bannerPermalinksRepository.getBidsForPermalinks(shard, permalinks);
    }

    @Override
    protected Map<Long, PermalinksInfo> getResources(int shard,
                                                     List<BaseBannerWithResourcesForBsExport> bannersFromDb) {
        var bidToBannerMap = bannersFromDb.stream()
                .filter(this::hasReadyResource)
                .collect(toMap(BaseBannerWithResourcesForBsExport::getId, banner -> banner));

        var bids = bidToBannerMap.keySet();
        var bidToBannerPermalinksMap = StreamEx
                .of(Iterables.partition(bids, PERMALINKS_SELECT_CHUNK_SIZE).iterator())
                .flatCollection(permalinksChunk -> bannerPermalinksRepository.getPermalinks(shard, permalinksChunk))
                .groupingBy(BannerPermalink::getBannerId);
        return StreamEx.of(bidToBannerMap.keySet())
                .map(bid -> getPermalinkInfo(
                        bidToBannerPermalinksMap.getOrDefault(bid, List.of()),
                        bidToBannerMap.get(bid)))
                .collect(toMap(Pair::getLeft, Pair::getRight));
    }

    private int permalinkSortComparator(BannerPermalink permalink1, BannerPermalink permalink2) {
        int c = permalink1.getPermalinkId().compareTo(permalink2.getPermalinkId());
        if (c == 0) {
            c = permalink1.getChainId().compareTo(permalink2.getChainId());
        }
        return c;
    }

    private Pair<Long, PermalinksInfo> getPermalinkInfo(Collection<BannerPermalink> bannerPermalinks,
                                                        BaseBannerWithResourcesForBsExport banner) {
        BannerPermalink manualPermalink = StreamEx.of(bannerPermalinks)
                .filter(permalink -> MANUAL.equals(permalink.getPermalinkAssignType()))
                .findFirst()
                .orElse(null);
        BannerPermalink autoPermalink = StreamEx.of(bannerPermalinks)
                .filter(permalink -> AUTO.equals(permalink.getPermalinkAssignType()))
                .reverseSorted(this::permalinkSortComparator)
                .findFirst()
                .orElse(null);

        Long permalinkId = 0L;
        String href = "";
        String site = "";
        String domainFilter = "";
        PermalinkAssignType assignType = null;
        if (Objects.nonNull(manualPermalink) && PUBLISHED.equals(manualPermalink.getStatusPublish())) {
            permalinkId = manualPermalink.getPermalinkId();
            if (Boolean.TRUE.equals(manualPermalink.getPreferVcardOverPermalink())) {
                assignType = AUTO;
            } else {
                assignType = MANUAL;
                if (banner instanceof CpmGeoPinBanner) {
                    href = String.format(HREF_TEMPLATE_FOR_CPM_GEO_PIN, permalinkId);
                    site = SITE_FOR_CPM_GEO_PIN;
                    domainFilter = String.format(DOMAIN_FILTER_TEMPLATE_FOR_CPM_GEO_PIN, permalinkId);
                } else {
                    domainFilter = permalinkId.toString();
                }
            }
        } else if (Objects.nonNull(autoPermalink)) {
            permalinkId = autoPermalink.getPermalinkId();
            assignType = AUTO;
        }

        List<Long> chainIds = StreamEx.of(bannerPermalinks)
                .sorted(this::permalinkSortComparator)
                .map(BannerPermalink::getChainId)
                .filter(chainId -> Objects.nonNull(chainId) && chainId > 0L)
                .toList();

        var permalinkInfo = PermalinksInfo.builder()
                .withPermalinkId(permalinkId)
                .withPermalinkHref(href)
                .withPermalinkSite(site)
                .withPermalinkDomainFilter(domainFilter)
                .withPermalinkAssignType(assignType)
                .withPermalinkChainIds(chainIds)
                .build();
        return Pair.of(banner.getId(), permalinkInfo);
    }

    private boolean hasReadyResource(BaseBannerWithResourcesForBsExport resourceFromDb) {
        return BannerStatusModerate.YES.equals(resourceFromDb.getStatusModerate());
    }
}
