package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader

import one.util.streamex.StreamEx
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.banner.resources.PromoExtension
import ru.yandex.direct.core.entity.banner.model.BannerToSendPromoExtensionForBsExport
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.promoextension.PromoExtensionRepository
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate
import ru.yandex.direct.ess.common.utils.TablesEnum
import ru.yandex.direct.ess.logicobjects.bsexport.resources.AdditionalInfo
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.BannerResource
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.getAdditionalInfoList
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.getLoaderResult
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.getResourceInternal
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.HrefAndSiteService
import ru.yandex.direct.utils.DateTimeUtils
import java.time.ZoneId
import java.util.function.Function
import ru.yandex.adv.direct.banner.resources.PromoExtensionType
import ru.yandex.direct.core.entity.banner.model.BannerWithHref
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.cashback
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.discount
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.free
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.gift
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.installment
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.profit
import java.util.stream.Collectors

@Component
class BannerPromoExtensionLoader(
    bannerResourcesLoaderContext: BannerResourcesLoaderContext,
    private val promoExtensionRepository: PromoExtensionRepository,
    private val bannerRelationsRepository: BannerRelationsRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val hrefAndSiteService: HrefAndSiteService,
) : IBannerResourceLoader<PromoExtension?> {
    private val bannerTypedRepository = bannerResourcesLoaderContext.bannerTypedRepository
    private val bannerResourcesHelper = bannerResourcesLoaderContext.bannerResourcesHelper

    private val LOAD_BANNERS_CHUNK_SIZE = 500
    private val MSK_TIMEZONE = ZoneId.of(DateTimeUtils.MOSCOW_TIMEZONE)

    private fun getClassesToLoadFromDb() =
        listOf(BannerToSendPromoExtensionForBsExport::class.java, BannerWithHref::class.java)

    override fun loadResources(
        shard: Int,
        bannerResourcesObjects: Collection<BsExportBannerResourcesObject>
    ): LoaderResult<PromoExtension?> {
        val bannerChangesToDbBannersWithObjectsMap = getBidsToDbBannerWithObject(shard, bannerResourcesObjects)

        val existingResources: List<BannerResource<PromoExtension?>> =
            getExistingResources(shard, bannerChangesToDbBannersWithObjectsMap)

        return getLoaderResult(bannerResourcesObjects, existingResources)
    }

    private fun getBidsToDbBannerWithObject(
        shard: Int,
        objects: Collection<BsExportBannerResourcesObject>
    ): Map<ChangeInfo, BannerFromDbWithLogicObject<BannerToSendPromoExtensionForBsExport>> {
        val additionalInfoList: List<AdditionalInfo> = getAdditionalInfoList(objects)
        val bannerChangesToLoad = getAdditionalBids(shard, additionalInfoList)

        val banners: List<BannerToSendPromoExtensionForBsExport> =
            StreamEx.ofSubLists(bannerChangesToLoad, LOAD_BANNERS_CHUNK_SIZE)
                .map { bannerChangesChunk: List<ChangeInfo> ->
                    bannerTypedRepository.getSafely(
                        shard,
                        bannerChangesChunk.map { it.id },
                        getClassesToLoadFromDb()
                    ).filterIsInstance<BannerToSendPromoExtensionForBsExport>()
                }.toFlatList(Function.identity())

        val bidsToBannerWithLogicObject = bannerResourcesHelper.getBidsToDbBanner(shard, banners)
        return bannerChangesToLoad
            .filter { bidsToBannerWithLogicObject.containsKey(it.id) }
            .associateWith { bidsToBannerWithLogicObject[it.id]!! }
    }

    private fun getExistingResources(
        shard: Int,
        bannerChangesToBannerWithLogicObject: Map<ChangeInfo, BannerFromDbWithLogicObject<BannerToSendPromoExtensionForBsExport>>
    ): List<BannerResource<PromoExtension?>> {
        if (bannerChangesToBannerWithLogicObject.isEmpty()) {
            return listOf()
        }

        val bidsToResources: Map<Long, PromoExtension?> = getResources(shard, bannerChangesToBannerWithLogicObject)

        val bidsToBannerWithLogicObject = bannerChangesToBannerWithLogicObject.mapKeys { it.key.id }
        return bidsToResources
            .map { (key, value): Map.Entry<Long, PromoExtension?> ->
                getResourceInternal(
                    bidsToBannerWithLogicObject[key]!!, value
                )
            }
    }

    /**
     * Если было изменение в объекете, не содержищим bid, метод возвращает
     */
    private fun getAdditionalBids(shard: Int, objects: Collection<AdditionalInfo>): List<ChangeInfo> {
        if (objects.isEmpty()) {
            return listOf()
        }
        val cids = objects
            .filter { it.additionalTable == TablesEnum.CAMPAIGN_PROMOACTIONS }
            .map { it.additionalId }
            .toMutableSet()
        val promoExtensionIds = objects
            .filter { it.additionalTable == TablesEnum.PROMOACTIONS }
            .map { it.additionalId }
            .toSet()
        val cidsByInsertedBids = objects
            .filter { it.additionalTable == TablesEnum.BANNERS }
            .map { it.additionalId }
            .let { bannerRelationsRepository.getCampaignIdsByBannerIds(shard, it.toSet()) }

        val cidsOfInsertedBidsHavingPromoExtension = promoExtensionRepository.getPromoExtensionIdsByCids(
            shard, cidsByInsertedBids.values.toSet()
        ).keys
        val cidsAffectedByPromoExtensionsChanges =
            promoExtensionRepository.getCidsByPromoExtensionIds(shard, promoExtensionIds).flatMap { it.value }

        val bidsResult = cidsByInsertedBids
            .filterValues { cidsOfInsertedBidsHavingPromoExtension.contains(it) }
            .map { ChangeInfo(it.key, ChangeInfo.ChangeType.NOT_ONLY_PROMO_CHANGED) }
            .toMutableSet()
        bidsResult.addAll(
            bannerRelationsRepository.getBannerIdsByCampaignIds(shard, cids)
                .map { ChangeInfo(it, ChangeInfo.ChangeType.NOT_ONLY_PROMO_CHANGED) })
        bidsResult.addAll(
            bannerRelationsRepository.getBannerIdsByCampaignIds(
                shard,
                cidsAffectedByPromoExtensionsChanges
            ).map { ChangeInfo(it, ChangeInfo.ChangeType.ONLY_PROMO_CHANGED) })
        return bidsResult.toList()
    }

    private fun getResources(
        shard: Int,
        bannerChangesToBannerWithLogicObject: Map<ChangeInfo, BannerFromDbWithLogicObject<BannerToSendPromoExtensionForBsExport>>
    ): Map<Long, PromoExtension?> {
        val bannerWithResourceFromDb: List<BannerToSendPromoExtensionForBsExport> =
            bannerChangesToBannerWithLogicObject.values
                .map { bannerWithObject: BannerFromDbWithLogicObject<BannerToSendPromoExtensionForBsExport> -> bannerWithObject.bannerFromDb }
        val bidsToChangeType = bannerChangesToBannerWithLogicObject.keys.stream()
            .collect(Collectors.toMap({ it.id }, { it.type }, { a, b -> mergeChangeInfo(a, b) }))
        val promoExtensionsByCid = promoExtensionRepository.getPromoExtensionsByCids(
            shard,
            bannerWithResourceFromDb.map { it.campaignId }.toSet()
        )

        val promoExtensionByBid: Map<Long, ru.yandex.direct.core.entity.promoextension.model.PromoExtension?> =
            bannerWithResourceFromDb
                .associate { banner -> banner.id to promoExtensionsByCid[banner.campaignId] }

        /**
         * Внутри extractCorrectHrefsByBid меняем баннер - должно идти последним действием, далее безопасно можно брать только id
         * Правильно хранить тип в BannerToSendPromoExtensionForBsExport и по нему строить копию, поправится в DIRECT-158076
         * Ну или не по типу, а классу баннера
         */
        val hrefByBannerId = extractCorrectHrefsByBid(shard, bannerWithResourceFromDb, promoExtensionByBid)

        val result: MutableMap<Long, PromoExtension?> = mutableMapOf()
        bannerWithResourceFromDb
            .map { it.id }
            .forEach { bid ->
                val currentPromoExtension = promoExtensionByBid[bid]
                if (currentPromoExtension?.statusModerate == PromoactionsStatusmoderate.Yes) {
                    result[bid] = toBsPromoExtension(currentPromoExtension, hrefByBannerId[bid])
                } else if (currentPromoExtension == null || currentPromoExtension.statusModerate == PromoactionsStatusmoderate.No ||
                    bidsToChangeType[bid] == ChangeInfo.ChangeType.NOT_ONLY_PROMO_CHANGED
                ) {
                    result[bid] = null
                }
            }
        return result
    }

    private fun mergeChangeInfo(a: ChangeInfo.ChangeType, b: ChangeInfo.ChangeType): ChangeInfo.ChangeType {
        return if (a == ChangeInfo.ChangeType.NOT_ONLY_PROMO_CHANGED) a else b
    }

    private fun extractCorrectHrefsByBid(
        shard: Int,
        banners: Collection<BannerToSendPromoExtensionForBsExport>,
        promoExtensionByBid: Map<Long, ru.yandex.direct.core.entity.promoextension.model.PromoExtension?>,
    ): Map<Long, String?> {
        val bannersForHrefExtraction =
            banners.filter { promoExtensionByBid[it.id] != null } //чтобы не извлекать для лишнего
        val campaignsByCid = campaignTypedRepository
            .getSafely(shard, bannersForHrefExtraction.map { it.campaignId }.toSet(), CommonCampaign::class.java)
            .associateBy { it.id }
        return bannersForHrefExtraction.associate { banner ->
            banner.id to extractHrefForSingleBanner(
                banner,
                promoExtensionByBid[banner.id]!!, //на null проверили выше
                campaignsByCid[banner.campaignId]
            )
        }
    }

    private fun extractHrefForSingleBanner(
        banner: BannerToSendPromoExtensionForBsExport,
        bannerPromoExtension: ru.yandex.direct.core.entity.promoextension.model.PromoExtension,
        campaign: CommonCampaign?,
    ): String? {
        if (campaign == null) {//надеюсь, что невозможно, но могут быть гонки с удалением кампании
            return null
        }
        if (banner is BannerWithHref) {
            val href =
                (if (bannerPromoExtension.href != null) bannerPromoExtension.href else banner.href) ?: return null
            banner.href = href
            banner.domain = null
            banner.domainId = null
            return hrefAndSiteService.extract(banner, campaign).href
        } else if (bannerPromoExtension.href != null) {
            return hrefAndSiteService.extract(bannerPromoExtension.href, banner, campaign).href
        } else {
            return null
        }
    }

    private fun toBsPromoExtension(
        promoExtension: ru.yandex.direct.core.entity.promoextension.model.PromoExtension,
        correctHrefForBannerPromo: String?,
    ) = PromoExtension.newBuilder()
        .setPromoExtensionId(promoExtension.id!!)
        .apply { if (correctHrefForBannerPromo != null) href = correctHrefForBannerPromo }
        .setDescription(promoExtension.compoundDescription)
        .apply {
            if (promoExtension.startDate != null) startTime = promoExtension.startDate!!
                .atStartOfDay(MSK_TIMEZONE).toInstant().epochSecond
        }
        .apply {
            if (promoExtension.finishDate != null) finishTime = promoExtension.finishDate!!
                .plusDays(1).atStartOfDay(MSK_TIMEZONE).minusSeconds(1).toInstant().epochSecond
        }
        .setType(corePromoExtensionTypeToBsType(promoExtension.type))
        .build()

}

fun corePromoExtensionTypeToBsType(type: PromoactionsType) =
    when (type) {
        discount -> PromoExtensionType.DISCOUNT
        profit -> PromoExtensionType.PROFIT
        cashback -> PromoExtensionType.CASHBACK
        gift -> PromoExtensionType.GIFT
        free -> PromoExtensionType.FREE
        installment -> PromoExtensionType.INSTALLMENT
    }.number

data class ChangeInfo(val id: Long, val type: ChangeType) {
    enum class ChangeType {
        ONLY_PROMO_CHANGED,
        NOT_ONLY_PROMO_CHANGED // если возникли еще изменения в привязке промоакции к кампании или в баннере
    }
}
