package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLandingForBsExport;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.TurboLandingInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.BsHrefParametrizingService;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.ReplacingParams;
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.CampaignNameTransliterator;
import ru.yandex.direct.utils.model.UrlParts;
import ru.yandex.direct.validation.constraint.StringConstraints;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.banner.model.BannerTurboLandingStatusModerate.NO;
import static ru.yandex.direct.core.entity.banner.model.BannerTurboLandingStatusModerate.YES;
import static ru.yandex.direct.utils.UrlUtils.trimPort;

@Component
public class BannerTurboLandingsLoader extends BaseBannerResourcesLoader<BannerWithTurboLandingForBsExport,
        TurboLandingInfo> {
    private static final Logger logger = LoggerFactory.getLogger(BannerTurboLandingsLoader.class);

    private static final String TURBO_LANDINGS_FILTER_DOMAIN_TEMPLATE = "%d.y-turbo";
    private final TurboLandingRepository turboLandingRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final BsHrefParametrizingService parametrizingService;
    private final CampaignNameTransliterator campaignNameTransliterator;

    public BannerTurboLandingsLoader(BannerResourcesLoaderContext context,
                                     TurboLandingRepository turboLandingRepository,
                                     CampaignTypedRepository campaignTypedRepository,
                                     BsHrefParametrizingService parametrizingService,
                                     CampaignNameTransliterator campaignNameTransliterator) {
        super(context);
        this.turboLandingRepository = turboLandingRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.parametrizingService = parametrizingService;
        this.campaignNameTransliterator = campaignNameTransliterator;
    }

    @Override
    public Class<BannerWithTurboLandingForBsExport> getClassToLoadFromDb() {
        return BannerWithTurboLandingForBsExport.class;
    }

    @Override
    protected Map<Long, TurboLandingInfo> getResources(int shard,
                                                       List<BannerWithTurboLandingForBsExport> bannersFromDb) {
        try {
            Map<Long, TurboLandingInfo> turboLandingsInfoMap = new HashMap<>();

            bannersFromDb.stream()
                    .filter(banner -> NO.equals(banner.getTurboLandingStatusModerate()))
                    .forEach(banner -> turboLandingsInfoMap.put(banner.getId(), null));

            var bannerWithTurboLandingModerated = bannersFromDb.stream()
                    .filter(banner -> YES.equals(banner.getTurboLandingStatusModerate()))
                    .collect(toList());

            turboLandingsInfoMap.putAll(getResourcesForModeratedTurboLandings(shard, bannerWithTurboLandingModerated));
            return turboLandingsInfoMap;
            // Для первого запуска, чтобы не останавливать транспорт целиком, попробуем завернуть загрузку в try-catch
            // уберем тут https://st.yandex-team.ru/DIRECT-130967
        } catch (RuntimeException e) {
            var bids = bannersFromDb.stream().map(BannerWithTurboLandingForBsExport::getId).collect(toList());
            logger.error("Failed to handle hrefs for bids " + bids, e);
            return Map.of();
        }
    }

    private Map<Long, TurboLandingInfo> getResourcesForModeratedTurboLandings(int shard,
                                                                              List<BannerWithTurboLandingForBsExport> banners) {
        var campaignIdToCampaigns = getCampaignIdToCampaignsMap(shard, banners);

        var turboLandingIds =
                banners.stream().map(BannerWithTurboLandingForBsExport::getTurboLandingId).collect(toList());
        var turbolandingIdToTurboLanding = turboLandingRepository.getTurboLandings(shard, turboLandingIds);
        return banners.stream()
                .filter(banner -> Objects.nonNull(banner.getTurboLandingId()))
                .filter(banner -> campaignIdToCampaigns.containsKey(banner.getCampaignId()))
                .filter(banner -> turbolandingIdToTurboLanding.containsKey(banner.getTurboLandingId()))
                .collect(toMap(BannerWithTurboLandingForBsExport::getId,
                        banner -> {
                            var campaign = campaignIdToCampaigns.get(banner.getCampaignId());
                            var turbolanding = turbolandingIdToTurboLanding.get(banner.getTurboLandingId());
                            return getTurboLandingInfo(banner, campaign, turbolanding);

                        }));
    }

    private TurboLandingInfo getTurboLandingInfo(BannerWithTurboLandingForBsExport banner, CommonCampaign campaign,
                                                 TurboLanding turboLanding) {
        var urlParts = getUrlParts(banner, campaign, turboLanding);
        var url = Objects.nonNull(urlParts) ? urlParts.toUrl() : "";
        var site = Objects.nonNull(urlParts) ? trimPort(urlParts.getDomain()) : "";
        return TurboLandingInfo.builder()
                .withTurbolandingId(banner.getTurboLandingId())
                .withHref(url)
                .withSite(site)
                .withDomainFilter(String.format(TURBO_LANDINGS_FILTER_DOMAIN_TEMPLATE,
                        banner.getTurboLandingId()))
                .build();
    }

    @Nullable
    private UrlParts getUrlParts(BannerWithTurboLandingForBsExport banner, CommonCampaign campaign,
                                 TurboLanding turboLanding) {
        if (!isValidHref(turboLanding.getUrl())) {
            return null;
        }
        var urlParts = UrlParts.fromUrl(turboLanding.getUrl());
        if (Objects.isNull(banner.getTurboLandingHrefParams()) || "".equals(banner.getTurboLandingHrefParams())) {
            return urlParts;
        }

        var turbolandingParams = banner.getTurboLandingHrefParams();
        turbolandingParams = StringUtils.removeStart(turbolandingParams, "/");
        turbolandingParams = StringUtils.removeStart(turbolandingParams, "?");
        turbolandingParams = StringUtils.removeStart(turbolandingParams, "&");
        var campaignName = Objects.nonNull(campaign.getName()) ? campaign.getName() : "";
        var builder = ReplacingParams.builder()
                .withBid(banner.getId())
                .withPid(banner.getAdGroupId())
                .withCid(banner.getCampaignId())
                .withCampaignType(campaign.getType())
                .withCampaignName(campaignName)
                .withCampaignNameLat(campaignNameTransliterator.translit(campaignName))
                .withCampaignCurrency(campaign.getCurrency());
        if (campaign instanceof CampaignWithStrategy) {
            builder = builder.withCampaignStrategy(((CampaignWithStrategy) campaign).getStrategy());
        }
        var replacingParams = builder.build();
        var parametrizedParams = parametrizingService.parameterize(turbolandingParams, replacingParams);
        return UrlParts.fromUrl(turboLanding.getUrl()).toBuilder().addParameters(parametrizedParams).build();
    }

    private Map<Long, CommonCampaign> getCampaignIdToCampaignsMap(int shard,
                                                                  Collection<BannerWithTurboLandingForBsExport> banners) {
        var campaignIds = banners.stream()
                .map(BannerWithTurboLandingForBsExport::getCampaignId)
                .collect(toList());

        return campaignTypedRepository.getSafely(shard, campaignIds, CommonCampaign.class).stream()
                .collect(toMap(CommonCampaign::getId, campaign -> campaign));
    }

    private boolean isValidHref(String href) {
        return Objects.nonNull(href) && !"".equals(href) && StringConstraints.isValidHref(href);
    }
}
