package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.bsexport.repository.resources.BsExportBannerVcardsRepository;
import ru.yandex.direct.core.bsexport.resources.model.Vcard;
import ru.yandex.direct.core.entity.banner.model.BannerVcardStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcardForBsExport;
import ru.yandex.direct.core.entity.banner.model.BaseBannerWithResourcesForBsExport;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.VcardInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.DomainFilterService;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

@Component
public class BannerVcardLoader extends BaseBannerResourcesLoader<BannerWithVcardForBsExport, VcardInfo> {
    private static final Logger logger = LoggerFactory.getLogger(BannerVcardLoader.class);
    private static final Pattern NOT_NUMBERS = Pattern.compile("[^\\d]");

    private final BsExportBannerVcardsRepository vcardsRepository;
    private final DomainFilterService domainFilterService;

    public BannerVcardLoader(BannerResourcesLoaderContext context,
                             BsExportBannerVcardsRepository vcardsRepository, DomainFilterService domainFilterService) {
        super(context);
        this.vcardsRepository = vcardsRepository;
        this.domainFilterService = domainFilterService;
    }

    @Override
    protected Class<BannerWithVcardForBsExport> getClassToLoadFromDb() {
        return BannerWithVcardForBsExport.class;
    }

    @Override
    protected Map<Long, VcardInfo> getResources(int shard,
                                                List<BannerWithVcardForBsExport> bannersFromDb) {
        try {
            var bannersWithVcardRejectedByModerationOrDeleted = StreamEx.of(bannersFromDb)
                    .filter(b -> BannerVcardStatusModerate.NO.equals(b.getVcardStatusModerate())
                            || Objects.isNull(b.getVcardId()))
                    .toMap(BannerWithVcardForBsExport::getId, b -> getVcardInfoForDeletedVcard());

            var bannersToCalculate = StreamEx.of(bannersFromDb)
                    .filter(b -> BannerVcardStatusModerate.YES.equals(b.getVcardStatusModerate())
                            && Objects.nonNull(b.getVcardId()))
                    .toList();

            var vcardsMap = getVcardsMap(shard, bannersToCalculate);

            var vcardIdToPhoneDomainFilterMap = getVcardPhoneDomainFilters(vcardsMap.values());

            return StreamEx.of(bannersToCalculate)
                    .filter(b -> vcardsMap.containsKey(b.getVcardId()))
                    .mapToEntry(BaseBannerWithResourcesForBsExport::getId,
                            banner -> {
                                var phoneDomainFilter = vcardIdToPhoneDomainFilterMap.get(banner.getVcardId());
                                return getVcardInfo(phoneDomainFilter);
                            })
                    .append(bannersWithVcardRejectedByModerationOrDeleted)
                    .toMap();

            // Для первого запуска, чтобы не останавливать транспорт целиком, попробуем завернуть загрузку в try-catch
            // уберем тут https://st.yandex-team.ru/DIRECT-130967
        } catch (RuntimeException e) {
            var bids = bannersFromDb.stream().map(BaseBannerWithResourcesForBsExport::getId).collect(toList());
            logger.error("Failed to handle vcards for bids " + bids, e);
            return Map.of();
        }
    }

    private Map<Long, Vcard> getVcardsMap(int shard, Collection<BannerWithVcardForBsExport> banners) {
        var vcardsToLoad = banners.stream()
                .filter(b -> Objects.nonNull(b.getVcardId()))
                .map(BannerWithVcardForBsExport::getVcardId)
                .collect(toSet());
        return vcardsRepository.getVcards(shard, vcardsToLoad);
    }

    private Map<Long, String> getVcardPhoneDomainFilters(Collection<Vcard> vcards) {
        var phoneDomains = StreamEx.of(vcards)
                .filter(v -> Objects.nonNull(v.getPhone()))
                .toMap(Vcard::getVcardId, v -> getPhoneDomain(v.getPhone()));

        var phoneDomainsSet = Set.copyOf(phoneDomains.values());
        var domainFilters = domainFilterService.getDomainsFilters(phoneDomainsSet);
        return StreamEx.of(vcards)
                .mapToEntry(Vcard::getVcardId, v -> v)
                .mapValues(v -> phoneDomains.get(v.getVcardId()))
                .mapValues(phoneDomain -> Objects.isNull(phoneDomain) ? "" : domainFilters.get(phoneDomain))
                .toMap();
    }


    private String getPhoneDomain(Phone phone) {
        var phoneStr = phone.getCountryCode() + phone.getCityCode() + phone.getPhoneNumber();
        phoneStr = NOT_NUMBERS.matcher(phoneStr).replaceAll("");
        return phoneStr + ".phone";
    }

    private VcardInfo getVcardInfo(String domainFilter) {
        return VcardInfo.builder()
                .withDomainFilter(domainFilter)
                .build();
    }

    private VcardInfo getVcardInfoForDeletedVcard() {
        return VcardInfo.builder().withDomainFilter("").build();
    }
}
