package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.AdditionalInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.container.BannerResource;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.BannerResourcesLoaderUtilsKt;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.BannerResourcesLoaderUtilsKt.getAdditionalInfoList;
import static ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.BannerResourcesLoaderUtilsKt.getLoaderResult;
import static ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.BannerResourcesLoaderUtilsKt.getResourceInternal;

public abstract class BaseBannerResourcesLoader<T extends BannerWithSystemFields, R> implements IBannerResourceLoader<R> {

    private static final int LOAD_BANNERS_CHUNK_SIZE = 10_000;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerResourcesHelper bannerResourcesHelper;

    BaseBannerResourcesLoader(BannerResourcesLoaderContext context) {
        this.bannerTypedRepository = context.getBannerTypedRepository();
        this.bannerResourcesHelper = context.getBannerResourcesHelper();
    }

    protected abstract Class<T> getClassToLoadFromDb();

    protected abstract Map<Long, R> getResources(int shard, List<T> bannerWithResourceFromDb);

    /**
     * Если было изменение в объекете, не содержищим bid, метод возвращает
     */
    List<Long> getAdditionalBids(int shard, Collection<AdditionalInfo> objects) {
        return List.of();
    }

    @Override
    public LoaderResult<R> loadResources(int shard,
                                         Collection<BsExportBannerResourcesObject> bannerResourcesObjects) {

        var bidsToDbBannersWithObjectsMap = getBidsToDbBannerWithObject(shard, bannerResourcesObjects);

        var deletedResources = bidsToDbBannersWithObjectsMap.values().stream()
                .filter(BannerFromDbWithLogicObject::isDeleted)
                .map(bannerWithObject ->
                        BannerResourcesLoaderUtilsKt.<T, R>getResourceInternal(bannerWithObject, null)
                )
                .collect(toList());

        List<BannerResource<R>> existingResources;
        if (!bidsToDbBannersWithObjectsMap.isEmpty()) {
            existingResources = getExistingResources(shard, bidsToDbBannersWithObjectsMap);
        } else {
            existingResources = List.of();
        }

        List<BannerResource<R>> resources = new ArrayList<>();

        resources.addAll(existingResources);
        resources.addAll(deletedResources);

        return getLoaderResult(bannerResourcesObjects, resources);
    }

    Map<Long, BannerFromDbWithLogicObject<T>> getBidsToDbBannerWithObject(int shard,
                                                                          Collection<BsExportBannerResourcesObject> objects) {

        var bidsToObjectMap = objects.stream()
                .filter(o -> Objects.nonNull(o.getBid()))
                .collect(toMap(BsExportBannerResourcesObject::getBid, object -> object, this::mergeObjects));

        Set<Long> bidsToLoad = new HashSet<>(bidsToObjectMap.keySet());

        var additionalInfoList = getAdditionalInfoList(objects);
        if (!additionalInfoList.isEmpty()) {
            bidsToLoad.addAll(getAdditionalBids(shard, additionalInfoList));
        }

        List<T> banners = StreamEx.ofSubLists(List.copyOf(bidsToLoad), LOAD_BANNERS_CHUNK_SIZE)
                .map(bidsChunk -> bannerTypedRepository.getSafely(shard, bidsChunk, getClassToLoadFromDb()))
                .toFlatList(Function.identity());
        return bannerResourcesHelper.getBidsToDbBanner(shard, banners, bidsToObjectMap);
    }

    /**
     * Если объект удален из базы, то берем только объект с пометкой удален
     * Иначе берем любой
     */
    private BsExportBannerResourcesObject mergeObjects(BsExportBannerResourcesObject o1,
                                                       BsExportBannerResourcesObject o2) {
        if (o1.isDeleted()) {
            return o1;
        }
        if (o2.isDeleted()) {
            return o2;
        }
        return o1;
    }

    private List<BannerResource<R>> getExistingResources(int shard,
                                                         Map<Long, BannerFromDbWithLogicObject<T>> bidsToBannerWithLogicObject) {
        var bannerToLoadResource = bidsToBannerWithLogicObject.values().stream()
                .filter(bannerWithObject -> !bannerWithObject.isDeleted())
                .map(BannerFromDbWithLogicObject::getBannerFromDb)
                .collect(toList());

        var bidsToResources = getResources(shard, bannerToLoadResource);

        return bidsToResources.entrySet().stream()
                .map(e -> getResourceInternal(bidsToBannerWithLogicObject.get(e.getKey()), e.getValue()))
                .collect(toList());
    }
}
