package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils;

import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.assertj.core.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.mobilecontent.model.OsType;

@ParametersAreNonnullByDefault
public class MobileAppStoreHrefParser {
    private static final Logger logger = LoggerFactory.getLogger(MobileAppStoreHrefParser.class);

    public static final String DEFAULT_LANG = "ru";
    public static final String DEFAULT_REGION = "ru";

    private static final String ANDROID_LANG_PARAM = "hl";
    private static final String ANDROID_REGION_PARAM = "gl";
    private static final String APPLE_LANG_PARAM = "l";
    private static final Pattern APPLE_REGION_PATTERN =
            Pattern.compile("\\/+(([^\\/]+)\\/+)?app\\/+([^\\/]+\\/+)?(id\\d+)\\/*");

    private MobileAppStoreHrefParser() {
    }

    public static String parseLang(OsType osType, String storeHref) {
        if (Strings.isNullOrEmpty(storeHref)) {
            return DEFAULT_LANG;
        }

        String lang = null;
        try {
            if (osType == OsType.ANDROID) {
                lang = getUriParam(storeHref, ANDROID_LANG_PARAM);
            } else if (osType == OsType.IOS) {
                lang = getUriParam(storeHref, APPLE_LANG_PARAM);
            }
        } catch (Exception ex) {
            logger.warn("Got an error while parsing store href params", ex);
        }

        return lang == null ? DEFAULT_LANG : lang.toLowerCase(Locale.ROOT);
    }

    public static String parseRegion(OsType osType, String storeHref) {
        if (Strings.isNullOrEmpty(storeHref)) {
            return DEFAULT_REGION;
        }

        String region = null;
        try {
            if (osType == OsType.ANDROID) {
                region = getUriParam(storeHref, ANDROID_REGION_PARAM);
            } else if (osType == OsType.IOS) {
                region = getAppleRegion(storeHref);
            }
        } catch (Exception ex) {
            logger.warn("Got an error while parsing store href params", ex);
        }

        return region == null ? DEFAULT_REGION : region.toLowerCase(Locale.ROOT);
    }

    private static String getAppleRegion(String storeHref) {
        var path = URI.create(storeHref).getPath();
        var matcher = APPLE_REGION_PATTERN.matcher(path);
        return matcher.find() ? matcher.group(2) : null;
    }

    private static String getUriParam(String storeHref, String paramName) {
        List<NameValuePair> params = URLEncodedUtils.parse(URI.create(storeHref), StandardCharsets.UTF_8.name());
        var param = params.stream().filter(p -> paramName.equals(p.getName())).findFirst().orElse(null);
        return param == null ? null : param.getValue();
    }
}
