package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;

import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

/**
 * Сервис для получения фильтра домена
 */
@Service
public class DomainFilterService {
    private final DomainRepository domainRepository;
    private final HostingsHandler hostingsHandler;

    public DomainFilterService(DomainRepository domainRepository, HostingsHandler hostingsHandler) {
        this.domainRepository = domainRepository;
        this.hostingsHandler = hostingsHandler;
    }

    /**
     * Для заданных доменов определяет фильтр.
     * Для каждого домена находит главное зеркало, если зеркало не найдено, то обрезает один уровень домена и ищет
     * для него.
     * Обезать домен можно до тех пор, пока в нем есть dot или он не равен своему хостингу
     * Оригинал на perl - https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/MirrorsTools
     * .pm?rev=7523610#L107
     * Разница в том, что в перле фильтр считался для каждого домена отдельно, что не оптимально
     *
     * @param domains домены в кодировке ascii
     * @return словарь домен -> фильтр домена
     */
    public Map<String, String> getDomainsFilters(Collection<String> domains) {
        if (domains.isEmpty()) {
            return Map.of();
        }
        var domainsInfoMap = domains.stream()
                .map(domain -> new DomainInfo().setCalculatedDomain(domain).setHosting(hostingsHandler.getHosting(domain)))
                .collect(toMap(DomainInfo::getCalculatedDomain, domainInfo -> domainInfo));


        while (true) {
            /* берем домены, для которых еще не посчитан фильтр */
            var domainsForIteration = domainsInfoMap.values().stream()
                    .filter(domainInfo -> Objects.isNull(domainInfo.getDomainFilter()))
                    .map(DomainInfo::getCalculatedDomain)
                    .collect(toSet());
            if (domainsForIteration.isEmpty()) {
                break;
            }
            var mainMirrors = domainRepository.getMainMirrors(domainsForIteration);

            domainsInfoMap.forEach((domain, domainInfo) -> {
                if (Objects.nonNull(domainInfo.getDomainFilter())) {
                    return;
                }
                if (mainMirrors.containsKey(domainInfo.getCalculatedDomain())) {
                    /* если для домена рассчитался фильтр, то запишем его */
                    var mainMirror = mainMirrors.get(domainInfo.getCalculatedDomain());
                    var filter = calculateFilter(mainMirror, domain);
                    domainInfo.setDomainFilter(filter);
                } else if (canRemoveOneLevel(domainInfo)) {
                    /* Для домена не нашлось зеркала, но можем удалить еще один уровень и посчитать для него */
                    var previousCalculatedDomain = domainInfo.getCalculatedDomain();
                    var calculatedDomain = removeOneLevel(previousCalculatedDomain);
                    domainInfo.setCalculatedDomain(calculatedDomain);
                } else {
                    /* Для домена не нашлось зеркал, значит фильтром будет он сам */
                    domainInfo.setDomainFilter(calculateFilter(domain, domain));
                }
            });
        }
        return domainsInfoMap.entrySet()
                .stream().collect(toMap(Map.Entry::getKey, e -> e.getValue().getDomainFilter()));
    }


    /**
     * Возвращает фильтр, рассчитываемый из главного зеркала и домена, для которого считалось зеркало
     * Получает хостинг зеркала и обрезает www. Если полученный домен является зеркалос к яндексу, то возвращается
     * оригинальный домен
     */
    private String calculateFilter(String mainMirror, String originalDomain) {
        var mirrorHosting = hostingsHandler.getHosting(mainMirror);
        mirrorHosting = hostingsHandler.stripWww(mirrorHosting);
        // для Яндекса можно показывать параллельно несколько объявлений, т.к. проекты более-менее отдельные
        // при этом после склейки доменов может получиться, что не-яндексные домены являются зеркалами к
        // яндексовым такие надо по-прежнему фильтровать, поэтому подставляем исходный домен для тех
        // доменов, где главным получается yandex.ru
        if ("yandex.ru".equals(mirrorHosting) || "www.yandex.ru".equals(mirrorHosting)) {
            mirrorHosting = originalDomain;
        }
        return mirrorHosting;
    }


    private String removeOneLevel(String domain) {
        return StringUtils.substringAfter(domain, ".");
    }

    private boolean canRemoveOneLevel(DomainInfo domainInfo) {
        return !domainInfo.getCalculatedDomain().equals(domainInfo.getHosting()) && domainInfo.getCalculatedDomain().contains(".");
    }

    /**
     * Внутренняя структура для хранения результата рассчета фильтра
     */
    private static class DomainInfo {
        /* домен, для которого будет искаться зеркало */
        String calculatedDomain;
        /* хостинг домена */
        String hosting;
        /* полученный фильтр для calculatedDomain */
        String domainFilter = null;

        public String getCalculatedDomain() {
            return calculatedDomain;
        }

        public DomainInfo setCalculatedDomain(String calculatedDomain) {
            this.calculatedDomain = calculatedDomain;
            return this;
        }

        public String getHosting() {
            return hosting;
        }

        public DomainInfo setHosting(String hosting) {
            this.hosting = hosting;
            return this;
        }

        public String getDomainFilter() {
            return domainFilter;
        }

        public DomainInfo setDomainFilter(String domainFilter) {
            this.domainFilter = domainFilter;
            return this;
        }
    }
}
