package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href;

import java.net.IDN;
import java.util.Objects;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithCampaignId;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.BannerTextFormatter;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.BsHrefParametrizingService;
import ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.ReplacingParams;
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.CampaignNameTransliterator;
import ru.yandex.direct.utils.UrlUtils;
import ru.yandex.direct.utils.model.UrlParts;
import ru.yandex.direct.validation.constraint.StringConstraints;

import static java.util.regex.Pattern.UNICODE_CHARACTER_CLASS;
import static ru.yandex.direct.common.util.TextUtils.smartStrip;
import static ru.yandex.direct.common.util.TextUtils.spaceCleaner;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DISALLOW_CAMPAIGN_NAME_LETTERS_RE;
import static ru.yandex.direct.utils.UrlUtils.getProtocol;
import static ru.yandex.direct.utils.UrlUtils.trimProtocol;

@Component
public class HrefAndSiteService {

    private static final Logger logger = LoggerFactory.getLogger(HrefAndSiteService.class);
    private static final Pattern DOMAIN_PATTERN =
            Pattern.compile(StringConstraints.DOMAIN_PATTERN, UNICODE_CHARACTER_CLASS);
    private final BannerTextFormatter bannerTextFormatter;
    private final BsHrefParametrizingService bsHrefParametrizingService;
    private final CampaignNameTransliterator transliterator;

    public HrefAndSiteService(BannerTextFormatter bannerTextFormatter,
                              BsHrefParametrizingService bsHrefParametrizingService,
                              CampaignNameTransliterator transliterator) {
        this.bannerTextFormatter = bannerTextFormatter;
        this.bsHrefParametrizingService = bsHrefParametrizingService;
        this.transliterator = transliterator;
    }

    /**
     * Получает из баннера и кампании ссылку и домен
     * Делает подстановку параметров в url
     */
    public HrefAndSite extract(BannerWithHref banner, CommonCampaign campaign) {
        return extract(banner.getHref(), banner, campaign);
    }

    /**
     * Получает из баннера и кампании ссылку и домен
     * Делает подстановку параметров в url
     */
    public HrefAndSite extract(String href, Banner banner, CommonCampaign campaign) {
        href = bannerTextFormatter.format(prepareHref(href), banner);
        String asciiDomainWithPort = extractAsciiDomainWithPort(href);
        href = prepareHref(href, asciiDomainWithPort, banner, campaign);
        var site = (banner instanceof BannerWithHref) && isValidDomain(((BannerWithHref) banner).getDomain())
                ? ((BannerWithHref) banner).getDomain() : UrlUtils.trimPort(asciiDomainWithPort);
        String siteAscii = IDN.toASCII(site).toLowerCase();
        return new HrefAndSite(href, site, siteAscii);
    }

    /**
     * Разворачивает ссылку из банера, делая подстановку параметров в url
     */
    public String expandHref(String href, Banner banner, CommonCampaign campaign) {
        href = bannerTextFormatter.format(prepareHref(href), banner);
        String asciiDomainWithPort = extractAsciiDomainWithPort(href);
        return prepareHref(href, asciiDomainWithPort, banner, campaign);
    }

    private String extractAsciiDomainWithPort(String href) {
        String asciiDomainWithPort = "";
        if (isValidHref(href)) {
            var urlParts = UrlParts.fromUrl(href);
            asciiDomainWithPort = IDN.toASCII(urlParts.getDomain());
        }
        return asciiDomainWithPort;
    }

    private String prepareHref(String href, String asciiDomainWithPort, Banner banner, CommonCampaign campaign) {
        String resultHref = "";
        if (!isValidHref(href)) {
            logger.info("Not valid href {} for banner {}", href, banner.getId());
        } else {
            var urlParts = UrlParts.fromUrl(href);
            resultHref = urlParts.toBuilder().withDomain(asciiDomainWithPort).build().toUrl();
        }

        var replacingParams = getReplacingParams(banner, campaign);
        return bsHrefParametrizingService.parameterize(resultHref, replacingParams);
    }

    public boolean isValidHref(String href) {
        return Objects.nonNull(href) && !"".equals(href) && StringConstraints.isValidHref(href);
    }

    public boolean isValidDomain(String domain) {
        if (Objects.isNull(domain) || "".equals(domain)) {
            return false;
        }
        if (!DOMAIN_PATTERN.matcher(domain).matches()) {
            return false;
        }
        try {
            IDN.toASCII(domain);
        } catch (IllegalArgumentException e) {
            return false;
        }
        return true;
    }

    private ReplacingParams getReplacingParams(Banner banner, CommonCampaign campaign) {
        var campaignName = Objects.nonNull(campaign.getName()) ?
                campaign.getName().replaceAll(DISALLOW_CAMPAIGN_NAME_LETTERS_RE, "") : "";
        var campaignNameLat = transliterator.translit(campaignName);
        var result = ReplacingParams.builder()
                .withBid(banner.getId())
                .withCampaignType(campaign.getType())
                .withCampaignName(campaignName)
                .withCampaignNameLat(campaignNameLat)
                .withCampaignCurrency(campaign.getCurrency());
        if (banner instanceof BannerWithAdGroupId) {
            result = result.withPid(((BannerWithAdGroupId) banner).getAdGroupId());
        }
        if (banner instanceof BannerWithCampaignId) {
            result = result.withCid(((BannerWithCampaignId) banner).getCampaignId());
        }
        if (campaign instanceof CampaignWithStrategy) {
            result = result.withCampaignStrategy(((CampaignWithStrategy) campaign).getStrategy());
        }
        return result.build();
    }

    /**
     * Удаялет пробелы, заменяет различные дефисы на обычный, заменить различные одинарные и двойные кавычки на обычные
     * Подставляет протокол, если его не было
     */
    public String prepareHref(String href) {
        if (Objects.isNull(href)) {
            return "";
        }
        href = spaceCleaner(href);
        href = smartStrip(href);
        var protocol = getProtocol(href, "http");
        return protocol + "://" + trimProtocol(href);
    }
}
