package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.mobilecontent;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.utils.model.UrlParts;

/**
 * Преобразовывет ссылки в зависимости от трекера для приложений
 */
public abstract class BaseTrackerHrefHandler {

    private static final String IOS_IFA_REPLACEMENT = "{ios_ifa}";
    private static final String GOOGLE_AID_REPLACEMENT = "{google_aid}";
    private static final String LOGID_REPLACEMENT = "{logid}";
    protected static final String TRACKID_REPLACEMENT = "{trackid}";
    protected static final String OAID_REPLACEMENT = "{oaid}";

    private final Pattern trackerHrefPattern;
    private final Map<OsType, Map<String, String>> parametersMap = new EnumMap<>(OsType.class);

    public BaseTrackerHrefHandler() {
        var hrefPatterns = domainPatterns();
        this.trackerHrefPattern = Pattern.compile("(?:"
                + String.join("|", hrefPatterns)
                + ")");

        initParameters();
    }

    abstract List<String> domainPatterns();

    abstract void initParameters();

    /**
     * Добавляет параметры к ссылке, в зависимости от трекера для приложений, который получается из ссылки
     */
    public String handle(String href, OsType osType) {
        UrlParts hrefParts = UrlParts.fromUrl(href);
        if (!trackerHrefPattern.matcher(hrefParts.getDomain()).find()) {
            return null;
        }

        if (Objects.isNull(osType)) {
            return null;
        }

        List<Pair<String, String>> hrefParameters = replaceMacroParamValue(hrefParts.getParameters(), LOGID_REPLACEMENT, TRACKID_REPLACEMENT);
        var params = parametersMap.getOrDefault(osType, Map.of());
        var paramsForHref = getDependentParameters(osType, hrefParameters);
        var paramsToKeep = Objects.isNull(hrefParameters) ? null : hrefParameters.stream()
                .filter(param -> !params.containsKey(param.getKey()) && !paramsForHref.containsKey(param.getKey()))
                .collect(Collectors.toList());

        UrlParts.Builder hrefBuilder = hrefParts.toBuilder();
        hrefBuilder.withParameters(paramsToKeep).addParameters(params).addParameters(paramsForHref);
        return postProcessBuilder(hrefBuilder).build().toUrl();
    }

    /**
     * Добавить постоянный параметр parameter={ios_ofa} для os_type = IOS
     *
     * @param parameter имя параметра
     */
    void addPermanentIosIfaParameter(String parameter) {
        addPermanentOsSpecifiedParameter(OsType.IOS, parameter, IOS_IFA_REPLACEMENT);
    }

    /**
     * Добавить постоянный параметр parameter={google_aid} для os_type = ANDROID
     *
     * @param parameter имя параметра
     */
    void addPermanentGoogleAidParameter(String parameter) {
        addPermanentOsSpecifiedParameter(OsType.ANDROID, parameter, GOOGLE_AID_REPLACEMENT);
    }

    /**
     * Добавить постоянный параметр parameter={oaid} для os_type = ANDROID
     *
     * @param parameter имя параметра
     */
    void addPermanentOaidParameter(String parameter) {
        addPermanentOsSpecifiedParameter(OsType.ANDROID, parameter, OAID_REPLACEMENT);
    }

    /**
     * Добавить постоянный параметр parameter={trackid}
     *
     * @param parameter имя параметра
     */
    void addPermanentTrackIdParameter(String parameter) {
        addPermanentParameter(parameter, TRACKID_REPLACEMENT);
    }

    /**
     * Добавить постоянный параметр parameter=value для опреденной ОС
     * Если в качестве значения нужно передать макрос, то его нужно явно обернуть в фигурные скобки
     * Если параметр с таким именем уже есть в ссылке, то его значение будет заменено
     *
     * @param parameter имя параметра
     */
    void addPermanentOsSpecifiedParameter(OsType osType, String parameter, String value) {
        parametersMap.computeIfAbsent(osType, os -> new HashMap<>()).put(parameter, value);
    }

    /**
     * Добавить постоянный параметр parameter=value
     * Если в качестве значения нужно передать макрос, то его нужно явно обернуть в фигурные скобки
     *
     * @param parameter имя параметра
     */
    void addPermanentParameter(String parameter, String value) {
        addPermanentOsSpecifiedParameter(OsType.IOS, parameter, value);
        addPermanentOsSpecifiedParameter(OsType.ANDROID, parameter, value);
    }

    /**
     * Получить параметры в зависимости от ссылки и типа ОС
     * Если для трекера нужны только перманентные параметры, то переопределять не нужно
     */
    Map<String, String> getDependentParameters(OsType osType, List<Pair<String, String>> hrefParameters) {
        return Map.of();
    }

    UrlParts.Builder postProcessBuilder(UrlParts.Builder builder) {
        return builder;
    }

    private boolean containsMacroParamValue(List<Pair<String, String>> parameters, String macro) {
        if (parameters == null || !macro.startsWith("{") || !macro.endsWith("}")) {
            return false;
        }
        return parameters.stream()
                .anyMatch(p -> p.getValue() != null && p.getValue().compareToIgnoreCase(macro) == 0);
    }

    private List<Pair<String, String>> replaceMacroParamValue(List<Pair<String, String>> parameters, String oldMacro, String newMacro) {
        if (parameters == null || !oldMacro.startsWith("{") || !oldMacro.endsWith("}") || !newMacro.startsWith("{") || !newMacro.endsWith("}")) {
            return parameters;
        }

        return parameters.stream()
                .map(p -> p.getValue() != null && p.getValue().compareToIgnoreCase(oldMacro) == 0 ? Pair.of(p.getKey(), newMacro) : p)
                .collect(Collectors.toList());
    }
}
