package ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.hrefs.parameterizer.CampaignTypeMapper;
import ru.yandex.direct.hrefs.parameterizer.HrefParameter;

import static java.util.Map.entry;
import static ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.ConversionResult.macro;
import static ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.ConversionResult.notFound;
import static ru.yandex.direct.logicprocessor.processors.bsexport.resources.loader.utils.href.parameterizer.ConversionResult.substitution;

/**
 * Преобразует токены в параметры
 * Для токена {@link HrefParameter#AD_ID} возвращает макрос, если кампания dynamic или performance, иначе подставляет
 * значение bid
 */
public class SubstitutionConverter {

    private static final Set<CampaignType> GENERATED_BANNERS = Set.of(CampaignType.DYNAMIC, CampaignType.PERFORMANCE);
    private static final int MAX_CAMPAIGN_NAME_LENGTH = 60;
    private final CampaignTypeMapper campaignTypeMapper;
    private final Map<HrefParameter, Function<ReplacingParams, ConversionResult>> bsSubstitutionsMap = Map.ofEntries(
            entry(HrefParameter.AD_ID, this::convertAdId),
            entry(HrefParameter.AD_GROUP_ID, params -> substitution(params.getPid().toString())),
            entry(HrefParameter.CAMPAIGN_ID, params -> substitution(params.getCid().toString())),
            entry(HrefParameter.CAMPAIGN_TYPE, this::convertCampaignType),
            entry(HrefParameter.CAMPAIGN_NAME, params -> {
                var truncatedName = truncateCampaignName(params.getCampaignName());
                return substitution(truncatedName);
            }),
            entry(HrefParameter.CAMPAIGN_NAME_LAT, params -> {
                var truncatedName = truncateCampaignName(params.getCampaignNameLat())
                        .replaceAll("[^a-zA-Z0-9_]", "_");
                return substitution(truncatedName);
            }),
            entry(HrefParameter.CAMPAIGN_CURRENCY, params -> {
                var currency = params.getCampaignCurrency();
                return currency == CurrencyCode.YND_FIXED ? notFound() : substitution(currency.toString());
            }),
            entry(HrefParameter.CAMPAIGN_CURRENCY_CODE, params -> {
                var currency = params.getCampaignCurrency();
                return currency == CurrencyCode.YND_FIXED ? notFound()
                        : substitution(Currencies.getCurrency(currency).getIsoNumCode().toString());
            }),
            entry(HrefParameter.CAMPAIGN_COST_TYPE, this::convertCampaignCostType),
            entry(HrefParameter.CAMPAIGN_COST, this::convertCampaignCost)
    );

    /**
     * Обрезает строку на первом слове, с которым длина строки станет не меньше
     * {@link SubstitutionConverter#MAX_CAMPAIGN_NAME_LENGTH} символов
     * Аналог https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/TextTools.pm?rev=r8176856#L495
     *
     * @param text строка
     * @return префикс этой строки, не длиннее {@link SubstitutionConverter#MAX_CAMPAIGN_NAME_LENGTH} символов
     */
    private static String truncateCampaignName(String text) {
        if (text.length() <= MAX_CAMPAIGN_NAME_LENGTH) {
            return text;
        }

        var substring = text.substring(0, MAX_CAMPAIGN_NAME_LENGTH);
        var lastWhiteSpaceIndex = substring.lastIndexOf(' ');
        if (lastWhiteSpaceIndex == -1) {
            return substring;
        } else {
            return substring.substring(0, lastWhiteSpaceIndex);
        }
    }

    private Map<String, Function<ReplacingParams, ConversionResult>> replacingFunctionMap;

    public SubstitutionConverter() {
        this.campaignTypeMapper = new CampaignTypeMapper();
        initReplacingFunctionMap();
    }

    public ConversionResult replace(String token, ReplacingParams replacingParams) {
        if (replacingFunctionMap.containsKey(token)) {
            return replacingFunctionMap.get(token).apply(replacingParams);
        }
        return null;
    }

    private void initReplacingFunctionMap() {
        replacingFunctionMap = EntryStream.of(bsSubstitutionsMap)
                .flatMapKeys(hrefParam -> hrefParam.getKeys().stream())
                .toMap();
    }


    private ConversionResult convertAdId(ReplacingParams params) {
        if (GENERATED_BANNERS.contains(params.getCampaignType())) {
            return macro("PEID");
        }

        return substitution(Objects.requireNonNull(params.getBid()).toString());
    }

    private ConversionResult convertCampaignType(ReplacingParams params) {

        if (!"".equals(campaignTypeMapper.map(params.getCampaignType()))) {
            return substitution(campaignTypeMapper.map(params.getCampaignType()));
        }
        throw new IllegalStateException("Unknown campaign type " + params.getCampaignType());
    }

    private ConversionResult convertCampaignCostType(ReplacingParams params) {
        var strategy = params.getCampaignStrategy();
        if (strategy == null) {
            return notFound();
        }
        if (strategy.isAvgCpi() && strategy.getStrategyData().getGoalId() == null) {
            return substitution("CPI");
        } else if (strategy.isCpc()) {
            return substitution("CPC");
        } else if (strategy.isCpm()) {
            return substitution("CPM");
        } else {
            return notFound();
        }
    }

    private ConversionResult convertCampaignCost(ReplacingParams params) {
        var strategy = params.getCampaignStrategy();
        if (strategy == null || strategy.getStrategyData() == null) {
            return notFound();
        }
        BigDecimal value = null;
        if (strategy.isAvgCpi() && strategy.getStrategyData().getGoalId() == null) {
            value = strategy.getStrategyData().getAvgCpi();
        } else if (strategy.isCpc()) {
            value = strategy.getStrategyData().getAvgBid();
        } else if (strategy.isCpm()) {
            value = strategy.getStrategyData().getAvgCpm();
        }
        return value == null ? notFound() : substitution(value.toString());
    }
}
