package ru.yandex.direct.logicprocessor.processors.bsexport.resync

import com.google.protobuf.Message
import org.slf4j.LoggerFactory
import ru.yandex.direct.bstransport.yt.repository.common.RowPosition
import ru.yandex.direct.bstransport.yt.service.BaseTableToQueueResyncService
import ru.yandex.direct.bstransport.yt.service.YtHashBorders
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.scheduler.support.DirectParameterizedJob
import ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.primitives.GaugeInt64
import java.time.Instant
import java.util.concurrent.CountDownLatch
import java.util.concurrent.TimeUnit

data class State<T : RowPosition>(
    val startTime: Long,
    val objectsCnt: Long,
    val position: T
)


/**
 * Джоба для переотправки ресурсов баннера из сортированной таблицы в очередь Casesar
 * Нужно для того, чтобы если на стороне Caesar что то случилось с баннером,
 * мы бы его гарантированно переотправили в течение 2х недель
 * Кроме того, она позволяет проверять корректность работы механизма переотправки.
 *
 * Во время тестирования в проде отключена
 */
abstract class BaseTableToQueueResyncJob<T1 : Message, T2 : RowPosition>(
    private val tableToQueueResyncService: BaseTableToQueueResyncService<T1, T2>,
    private val paramSource: TableToQueueParamSource)
    : DirectParameterizedJob<Int>() {

    companion object {
        private val logger = LoggerFactory.getLogger(BaseTableToQueueResyncJob::class.java)
    }

    abstract val chunkSizeProperty: PpcProperty<Int>
    abstract val relaxTimeProperty: PpcProperty<Int>
    abstract val metricsPrefix: String
    private val stopped: CountDownLatch = CountDownLatch(1)

    private lateinit var resyncSpeed: GaugeInt64
    private lateinit var resyncedObjects: GaugeInt64
    private lateinit var tableToQueueStateProp: PpcProperty<String>
    private var isInitialized = false
    protected lateinit var ytHashBorders: YtHashBorders

    private fun initialize() {
        ytHashBorders = tableToQueueResyncService.getYtHashBorders(param.toInt(), paramSource.chunksCnt)
        resyncSpeed = SOLOMON_REGISTRY.gaugeInt64("${metricsPrefix}_resync_speed", Labels.of("bucket_number", param))
        resyncedObjects = SOLOMON_REGISTRY.gaugeInt64("${metricsPrefix}_resync_cnt", Labels.of("bucket_number", param))
        tableToQueueStateProp = getTableToQueueStateProp(param)
        isInitialized = true
    }

    abstract fun getState(jsonState: String?): State<T2>

    abstract fun getTableToQueueStateProp(param: String): PpcProperty<String>
    override fun execute() {
        if (!isInitialized) {
            initialize()
        }
        if (ytHashBorders.isAbsent) {
            logger.warn("There is no work for batch $param")
            stopped.await(1, TimeUnit.MINUTES)
            return
        }
        val prevState = readState()

        logger.info("Iteration for param $param started. Position: ${prevState.position}")

        val limit = chunkSizeProperty.getOrDefault(100_000)
        val relaxTime = relaxTimeProperty.getOrDefault(300)
        val objectsWithPosition = tableToQueueResyncService
            .resyncObjects(prevState.position, limit, ytHashBorders.second)

        val resources = objectsWithPosition.objects

        val resourcesPosition = objectsWithPosition.position
        val newState = if (resources.size < limit) {
            logger.info("Job for param $param finished resync, " +
                "objectsCnt: ${prevState.objectsCnt + objectsWithPosition.objects.size}," +
                "duration: ${Instant.now().epochSecond - prevState.startTime} sec")
            // получить начальный стейт, если дочитали все строки
            getState(null)
        } else {
            State(
                prevState.startTime,
                prevState.objectsCnt + objectsWithPosition.objects.size,
                resourcesPosition)
        }

        saveState(newState)
        stopped.await(relaxTime.toLong(), TimeUnit.SECONDS)
        sendMetrics(newState)
        logger.info("Iteration for param $param finished")
    }

    private fun readState(): State<T2> {
        val stateJson = tableToQueueStateProp.get()
        return getState(stateJson)
    }

    private fun saveState(newState: State<T2>) {
        tableToQueueStateProp.set(JsonUtils.toJson(newState))
    }

    private fun sendMetrics(state: State<T2>) {
        resyncedObjects.set(state.objectsCnt)
        val speed = state.objectsCnt / (Instant.now().epochSecond - state.startTime)
        resyncSpeed.set(speed)

    }

    override fun finish() {
        super.finish()
        isInitialized = false
        stopped.countDown()
    }

}
