package ru.yandex.direct.logicprocessor.processors.bsexport.strategy

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.adv.direct.strategy.Strategy
import ru.yandex.direct.bstransport.yt.repository.strategy.StrategyYtRepository
import ru.yandex.direct.bstransport.yt.utils.CaesarIterIdGenerator
import ru.yandex.direct.common.log.container.bsexport.LogBsExportEssData
import ru.yandex.direct.common.log.service.LogBsExportEssService
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.ess.logicobjects.bsexport.strategy.BsExportStrategyObject
import ru.yandex.direct.ess.logicobjects.bsexport.strategy.StrategyResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.container.StrategyHandlerContainer
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.container.StrategyWithBuilder
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.handler.IStrategyResourceHandler
import java.time.Clock
import java.util.IdentityHashMap

@Service
class BsExportStrategyService(
    private val logBsExportEssService: LogBsExportEssService,
    private val caesarIterIdGenerator: CaesarIterIdGenerator,
    private val strategyTypedRepository: StrategyTypedRepository,
    private val clock: Clock = Clock.systemUTC(),
    private val resourceHandlers: List<IStrategyResourceHandler>,
    private val ytRepository: StrategyYtRepository,
    private val clientService: ClientService
) {
    private val resourceHandlersMap: Map<StrategyResourceType, IStrategyResourceHandler> = resourceHandlers
        .associateBy({ it.resourceType }) { it }

    fun processStrategies(shard: Int, logicObjects: List<BsExportStrategyObject>) {
        val iterId = caesarIterIdGenerator.generateCaesarIterId()
        if (logicObjects.isNotEmpty()) {
            modifyStrategies(shard, logicObjects, iterId)
        }
    }

    private fun modifyStrategies(shard: Int, updatedObjects: Collection<BsExportStrategyObject>, iterId: Long) {
        val strategiesWithBuildersById = loadStrategiesFromDb(shard, updatedObjects, iterId)

        val clientIdByStrategyId = strategiesWithBuildersById.values.map { it.strategy }
            .filterIsInstance<CommonStrategy>()
            .associate { it.id to ClientId.fromLong(it.clientId) }
        val clientById = clientService.getClients(shard, clientIdByStrategyId.values)
            .associateBy { it.id }

        handleStrategies(shard, clientById, strategiesWithBuildersById)

        try {
            modifyLoadedStrategies(strategiesWithBuildersById)
        } catch (e: Exception) {
            logger.error("Strategies processing failed for strategy ids ${strategiesWithBuildersById.keys}")
            throw e
        }
    }

    private fun loadStrategiesFromDb(
        shard: Int,
        updatedObjects: Collection<BsExportStrategyObject>,
        iterId: Long
    ): Map<Long, StrategyWithBuilder<BaseStrategy>> {
        val strategyIdsToLoad: Set<Long> = updatedObjects
            .flatMap { obj -> getHandlersToObjects(obj) }
            .groupByTo(IdentityHashMap(), { it.first }) { it.second }
            .flatMapTo(hashSetOf()) { (handler, objects) -> handler.getStrategyIdsToLoad(shard, objects).toSet() }

        val idToModelTyped = strategyTypedRepository.getIdToModelTyped(shard, strategyIdsToLoad)
        val updateTime = clock.instant().epochSecond

        return idToModelTyped.mapValues {
            StrategyWithBuilder(
                it.value,
                Strategy.newBuilder()
                    .setIterId(iterId)
                    .setStrategyId(it.key)
                    .setUpdateTime(updateTime)
            )
        }
    }

    private fun handleStrategies(
        shard: Int,
        clientById: Map<Long, Client>,
        strategiesWithBuildersById: Map<Long, StrategyWithBuilder<BaseStrategy>>
    ) {
        if (strategiesWithBuildersById.isNotEmpty()) {
            for (handler in resourceHandlers) {
                handler.handle(StrategyHandlerContainer(shard, clientById), strategiesWithBuildersById)
            }
        }
    }

    private fun getHandlersToObjects(obj: BsExportStrategyObject) =
        getHandlers(obj.strategyResourceType).map { handler -> handler to obj }

    private fun modifyLoadedStrategies(strategiesWithBuildersById: Map<Long, StrategyWithBuilder<BaseStrategy>>) {
        if (strategiesWithBuildersById.isEmpty()) {
            return
        }

        val strategies = strategiesWithBuildersById.map { (_, v) -> v.builder.build() }
        ytRepository.modify(strategies)
        logStrategies(strategies)
    }

    private fun logStrategies(strategies: List<Strategy>) {
        val recordsToLog = strategies
            .map {
                LogBsExportEssData<Strategy>()
                    .withStrategyId(it.strategyId)
                    .withData(it)
            }

        logBsExportEssService.logData(recordsToLog, DATA_TYPE)
    }

    private fun getHandlers(resourceType: StrategyResourceType) =
        when (resourceType) {
            StrategyResourceType.ALL -> resourceHandlersMap.values
            else -> resourceHandlersMap[resourceType]?.let { listOf(it) } ?: listOf()
        }

    companion object {
        private val logger = LoggerFactory.getLogger(BsExportStrategyService::class.java)
        private const val DATA_TYPE = "strategy"
    }
}
