package ru.yandex.direct.logicprocessor.processors.bsexport.strategy.handler

import org.springframework.stereotype.Component
import ru.yandex.adv.direct.strategy.AutobudgetRestart
import ru.yandex.adv.direct.strategy.Strategy
import ru.yandex.direct.autobudget.restart.repository.PackageStrategyAutobudgetRestartRepository
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.autobudget.restart.repository.StrategyRestartData
import ru.yandex.direct.autobudget.restart.service.StrategyState
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.ess.logicobjects.bsexport.strategy.StrategyResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.container.StrategyHandlerContainer
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.container.StrategyWithBuilder
import java.time.ZoneOffset

@Component
class StrategyAutobudgetRestartFieldsHandler(
    private val strategyAutobudgetRestartRepository: PackageStrategyAutobudgetRestartRepository,
    ppcPropertiesSupport: PpcPropertiesSupport
) : TypedStrategyResourceHandler<BaseStrategy>() {

    override val resourceType: StrategyResourceType = StrategyResourceType.AUTOBUDGET_RESTART
    override val strategiesToLoad: Class<BaseStrategy> = BaseStrategy::class.java

    private val autobudgetRestartSendingProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.PACKAGE_STRATEGY_AUTOBUDGET_RESTART_TRANSPORT_ENABLED)

    override fun fillExportObjects(
        container: StrategyHandlerContainer,
        strategyById: Map<Long, StrategyWithBuilder<BaseStrategy>>
    ) {
        if (!autobudgetRestartSendingProperty.getOrDefault(false)) {
            return
        }
        val restartData = readAutobudgetRestarts(container, strategyById.keys.toList())
        strategyById.forEach { (id, strategyWithBuilder) ->
            fillRestart(restartData[id], strategyWithBuilder.builder)
        }
    }

    private fun fillRestart(
        restartData: StrategyRestartData?,
        strategyBuilder: Strategy.Builder
    ) =
        restartData?.let {
            val autobudgetRestartObject = buildAutobudgetRestart(it.times, it.state)
            strategyBuilder.autobudgetRestart = autobudgetRestartObject
        }

    private fun readAutobudgetRestarts(
        container: StrategyHandlerContainer,
        strategyIds: List<Long>
    ): Map<Long, StrategyRestartData> =
        strategyAutobudgetRestartRepository.getAutobudgetRestartData(
            container.shard,
            strategyIds
        ).associateBy { it.strategyId }

    companion object {
        fun buildAutobudgetRestart(
            times: RestartTimes,
            state: StrategyState
        ): AutobudgetRestart {
            val builder = AutobudgetRestart.newBuilder()
            builder.restartTime = times.restartTime.toEpochSecond(ZoneOffset.UTC)
            builder.softRestartTime = times.softRestartTime.toEpochSecond(ZoneOffset.UTC)
            builder.restartReason = times.restartReason
            val stopTimestamp = state.stopTime?.toEpochSecond(ZoneOffset.UTC)

            if (stopTimestamp != null) {
                builder.stopTime = stopTimestamp
            }

            return builder.build()
        }
    }
}
