package ru.yandex.direct.logicprocessor.processors.bsexport.strategy.handler

import java.time.LocalTime
import java.time.ZoneOffset
import org.springframework.stereotype.Component
import ru.yandex.adv.direct.strategy.Strategy
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CPI_GOAL_ID
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.model.ClientNds
import ru.yandex.direct.core.entity.client.service.ClientNdsService
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpaPerFilter
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpcPerFilter
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpi
import ru.yandex.direct.core.entity.strategy.model.AutobudgetCrr
import ru.yandex.direct.core.entity.strategy.model.AutobudgetRoi
import ru.yandex.direct.core.entity.strategy.model.AutobudgetWeekBundle
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.model.StrategyAttributionModel
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgBid
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpa
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpm
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpv
import ru.yandex.direct.core.entity.strategy.model.StrategyWithBid
import ru.yandex.direct.core.entity.strategy.model.StrategyWithConversion
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCustomPeriodBudget
import ru.yandex.direct.core.entity.strategy.model.StrategyWithPayForConversion
import ru.yandex.direct.core.entity.strategy.model.StrategyWithWeeklyBudget
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.AUTOBUDGET_PROFITABILITY_DEFAULT
import ru.yandex.direct.currency.CurrencyCode
import ru.yandex.direct.currency.Money
import ru.yandex.direct.currency.Money.MICRO_MULTIPLIER_SCALE
import ru.yandex.direct.ess.logicobjects.bsexport.strategy.StrategyResourceType
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.container.StrategyHandlerContainer
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.container.StrategyWithBuilder
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.handler.StrategyCommonFieldsHandler.StrategyAttributionModelConverter.toExportAttributionModel
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.handler.StrategyCommonFieldsHandler.StrategyTypeConverter.toExportType
import ru.yandex.direct.logicprocessor.processors.bsexport.strategy.utils.ClientNdsConverter

@Component
class StrategyCommonFieldsHandler(
    private val clientNdsService: ClientNdsService,
    private val campaignsTypedRepository: CampaignTypedRepository
) : TypedStrategyResourceHandler<CommonStrategy>() {

    override val strategiesToLoad = CommonStrategy::class.java
    override val resourceType = StrategyResourceType.COMMON_FIELDS

    override fun fillExportObjects(
        container: StrategyHandlerContainer,
        strategyById: Map<Long, StrategyWithBuilder<CommonStrategy>>
    ) {
        val clientIdForNdsByStrategyId = strategyById.values
            .associate { it.strategy.id to getClient(container.clientById, it) }
            .mapValues { (_, client) -> getClientIdForNds(client) }
            .toMap()
        val ndsHistoryByClientId = clientNdsService.massGetClientNdsHistory(clientIdForNdsByStrategyId.values)
        val walletIds = strategyById.values.map { it.strategy.walletId }
        val walletById = getWalletById(container.shard, walletIds)

        strategyById.values.forEach {
            val client = getClient(container.clientById, it)
            fillCommon(
                client.agencyClientId,
                client.workCurrency,
                it.strategy,
                it.builder,
                ndsHistoryByClientId[clientIdForNdsByStrategyId[it.strategy.id]]?.toList(),
                walletById
            )
        }
    }

    private fun getClientIdForNds(client: Client): Long {
        if (client.agencyClientId != null && client.agencyClientId > 0 && true != client.nonResident) {
            return client.agencyClientId
        }
        return client.clientId
    }

    fun fillCommon(
        agencyId: Long?,
        currencyCode: CurrencyCode,
        strategy: CommonStrategy,
        builder: Strategy.Builder,
        ndsHistory: List<ClientNds>?,
        walletById: Map<Long, WalletTypedCampaign>
    ) {
        builder.walletExportId = strategy.walletId
        builder.type = toExportType(strategy.type).number
        builder.clientId = strategy.clientId
        builder.isArchived = strategy.statusArchived
        builder.attributionModel = toExportAttributionModel(strategy.attributionModel).number
        if (walletById.containsKey(strategy.walletId)) {
            val wallet = walletById[strategy.walletId]
            val walletOrderId = wallet!!.orderId
            builder.walletOrderId = if (walletOrderId == null || walletOrderId == 0L)
                wallet.id + BsOrderIdCalculator.ORDER_ID_OFFSET
            else wallet.orderId
        }
        if (currencyCode.currency.isoNumCode != null) {
            builder.currencyId = currencyCode.currency.isoNumCode.toLong()
        }
        if (agencyId != null) {
            builder.agencyId = agencyId
        }

        val convertedNds = ndsHistory
            ?.map { ClientNdsConverter.clientNdsToProto(it) }
            ?.toList() ?: listOf();

        builder.ndsHistoryBuilder.addAllNds(convertedNds)

        fillStrategyDataFields(strategy, builder, currencyCode)
    }

    object StrategyTypeConverter {
        private val coreToExportType: Map<StrategyName, ru.yandex.adv.direct.strategy.StrategyType> = mapOf(
            StrategyName.DEFAULT_ to ru.yandex.adv.direct.strategy.StrategyType.STRATEGY_TYPE_DEFAULT,
            //MIN_PRICE, NO_PREMIUM, AUTOBUDGET_WEEK_BUNDLE не актуальные типы
            StrategyName.AUTOBUDGET to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET,
            StrategyName.AUTOBUDGET_AVG_CLICK to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CLICK,
            StrategyName.AUTOBUDGET_AVG_CPA to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPA,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPA_PER_CAMP,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPA_PER_FILTER,
            StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPC_PER_CAMP,
            StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPC_PER_FILTER,
            StrategyName.AUTOBUDGET_AVG_CPI to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPI,
            StrategyName.AUTOBUDGET_AVG_CPV to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPV,
            StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_CRR to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_CRR,
            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_MAX_IMPRESSIONS,
            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_MAX_REACH to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_MAX_REACH,
            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_MEDIA to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_MEDIA,
            StrategyName.AUTOBUDGET_ROI to ru.yandex.adv.direct.strategy.StrategyType.AUTOBUDGET_ROI,
            StrategyName.CPM_DEFAULT to ru.yandex.adv.direct.strategy.StrategyType.CPM_DEFAULT,
            StrategyName.PERIOD_FIX_BID to ru.yandex.adv.direct.strategy.StrategyType.PERIOD_FIX_BID
        )

        fun toExportType(source: StrategyName): ru.yandex.adv.direct.strategy.StrategyType {
            return coreToExportType[source]
                ?: throw IllegalArgumentException("Unknown strategy type: $source")
        }
    }

    object StrategyAttributionModelConverter {
        private val coreToExportPlatform: Map<StrategyAttributionModel, ru.yandex.adv.direct.strategy.AttributionModel> =
            mapOf(
                StrategyAttributionModel.FIRST_CLICK to ru.yandex.adv.direct.strategy.AttributionModel.FIRST_CLICK,
                StrategyAttributionModel.FIRST_CLICK_CROSS_DEVICE to ru.yandex.adv.direct.strategy.AttributionModel.FIRST_CLICK_CROSS_DEVICE,
                StrategyAttributionModel.LAST_CLICK to ru.yandex.adv.direct.strategy.AttributionModel.LAST_CLICK,
                StrategyAttributionModel.LAST_SIGNIFICANT_CLICK to ru.yandex.adv.direct.strategy.AttributionModel.LAST_SIGNIFICANT_CLICK,
                StrategyAttributionModel.LAST_SIGNIFICANT_CLICK_CROSS_DEVICE to ru.yandex.adv.direct.strategy.AttributionModel.LAST_SIGNIFICANT_CLICK_CROSS_DEVICE,
                StrategyAttributionModel.LAST_YANDEX_DIRECT_CLICK to ru.yandex.adv.direct.strategy.AttributionModel.LAST_YANDEX_DIRECT_CLICK,
                StrategyAttributionModel.LAST_YANDEX_DIRECT_CLICK_CROSS_DEVICE to ru.yandex.adv.direct.strategy.AttributionModel.LAST_YANDEX_DIRECT_CLICK_CROSS_DEVICE
            )

        fun toExportAttributionModel(source: StrategyAttributionModel): ru.yandex.adv.direct.strategy.AttributionModel {
            return coreToExportPlatform[source]
                ?: throw IllegalArgumentException("Unknown strategy attribution model: $source")
        }
    }

    private fun fillStrategyDataFields(
        strategy: CommonStrategy,
        builder: Strategy.Builder,
        currencyCode: CurrencyCode
    ) {
        if (strategy is AutobudgetAvgCpaPerFilter && strategy.filterAvgCpa != null) {
            builder.avgCpa = Money.valueOf(strategy.filterAvgCpa, currencyCode).micros()
        }
        if (strategy is AutobudgetAvgCpcPerFilter && strategy.filterAvgBid != null) {
            builder.avgBid = Money.valueOf(strategy.filterAvgBid, currencyCode).micros()
        }
        if (strategy is AutobudgetAvgCpi) {
            if (strategy.avgCpi != null) {
                builder.avgCpa = Money.valueOf(strategy.avgCpi, currencyCode).micros()
            }
            if (strategy.goalId == null) {
                builder.goalId = DEFAULT_CPI_GOAL_ID
            }
        }
        if (strategy is AutobudgetCrr && strategy.crr != null) {
            builder.crr = strategy.crr
        }
        if (strategy is AutobudgetRoi && strategy.reserveReturn != null) {
            builder.roiCoef = strategy.roiCoef.scaleByPowerOfTen(MICRO_MULTIPLIER_SCALE).toLong()
            if (strategy.profitability != null) {
                builder.profitability = Money.valueOf(strategy.profitability, currencyCode).micros()
            } else {
                builder.profitability = AUTOBUDGET_PROFITABILITY_DEFAULT
            }
            builder.reserveReturn = strategy.reserveReturn
        }
        if (strategy is AutobudgetWeekBundle && strategy.limitClicks != null) {
            builder.limitClicks = strategy.limitClicks
        }
        if (strategy is StrategyWithAvgBid && strategy.avgBid != null) {
            builder.avgBid = Money.valueOf(strategy.avgBid, currencyCode).micros()
        }
        if (strategy is StrategyWithAvgCpa && strategy.avgCpa != null) {
            builder.avgCpa = Money.valueOf(strategy.avgCpa, currencyCode).micros()
        }
        if (strategy is StrategyWithAvgCpm && strategy.avgCpm != null) {
            builder.avgCpm = Money.valueOf(strategy.avgCpm, currencyCode).micros()
        }
        if (strategy is StrategyWithAvgCpv && strategy.avgCpv != null) {
            builder.avgCpv = Money.valueOf(strategy.avgCpv, currencyCode).micros()
        }
        if (strategy is StrategyWithBid && strategy.bid != null) {
            builder.bid = Money.valueOf(strategy.bid, currencyCode).micros()
        }
        if (strategy is StrategyWithConversion && strategy.goalId != null) {
            builder.goalId = strategy.goalId
        }
        if (strategy is StrategyWithPayForConversion && strategy.isPayForConversionEnabled != null) {
            builder.payForConversion = strategy.isPayForConversionEnabled
        }
        if (strategy is StrategyWithCustomPeriodBudget) {
            if (strategy.budget != null) {
                builder.customPeriodBudget = Money.valueOf(strategy.budget, currencyCode).micros()
            }
            if (strategy.start != null) {
                builder.start = strategy.start.toEpochSecond(LocalTime.MIN, ZoneOffset.UTC)
            }
            if (strategy.finish != null) {
                builder.finish = strategy.finish.toEpochSecond(LocalTime.MIN, ZoneOffset.UTC)
            }
            if (strategy.autoProlongation != null) {
                builder.autoProlongation = strategy.autoProlongation
            }
        }

        if (strategy is StrategyWithWeeklyBudget && strategy.sum != null) {
            builder.weekBudget = Money.valueOf(strategy.sum, currencyCode).micros()
        }
    }

    private fun getWalletById(shard: Int, cids: Collection<Long>): Map<Long, WalletTypedCampaign> {
        return campaignsTypedRepository.getTypedCampaignsMap(shard, cids)
            .filterValues { it is WalletTypedCampaign }
            .mapValues { it.value as WalletTypedCampaign };
    }
}
