package ru.yandex.direct.logicprocessor.processors.bsexport.utils

import org.jooq.Condition
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepositoryConstants
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.PHRASES
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.multitype.repository.filter.ConditionFilter
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory
import ru.yandex.direct.multitype.repository.filter.Filter

/**
 * [CampaignTypedRepository.getTyped] падает с исключением на кампаниях,
 * типа которых нет в [CampaignRepositoryConstants.CAMPAIGN_CLASS_TO_TYPE]
 * Поэтому методы этого репозитория загружают только кампании с поддерживаемыми типами
 */
@Service
class SupportedCampaignsService(
    private val campaignTypedRepository: CampaignTypedRepository,
    private val dslContextProvider: DslContextProvider,
) {

    fun getCampaignsByIdTyped(
        shard: Int,
        campaignIds: Collection<Long>,
    ): Map<Long, BaseCampaign> {
        val queryFilter = createQueryFilter(campaignIds)
        val resultCampaigns = campaignTypedRepository
            .getTypedCampaigns(
                dslContextProvider.ppc(shard),
                queryFilter,
                LimitOffset.maxLimited(),
            )
            .associateBy { it.id }
        if (campaignIds.size != resultCampaigns.size) {
            logger.error("Campaigns ${campaignIds.minus(resultCampaigns.keys)} was filtered")
        }
        return resultCampaigns
    }

    fun <T : BaseCampaign> getSafely(shard: Int, campaignIds: Collection<Long>, campaignClass: Class<T>): List<T> {
        val queryFilter = createQueryFilter(campaignIds)
        val resultCampaigns = campaignTypedRepository
            .getSafely(
                dslContextProvider.ppc(shard),
                queryFilter,
                campaignClass
            )
        if (campaignIds.size != resultCampaigns.size) {
            logger.warn("Campaigns ${campaignIds.minus(resultCampaigns.map { it.id })} was filtered")
        }
        return resultCampaigns
    }

    fun getClientsCampaignIds(shard: Int, clientIds: Collection<Long>): List<Long> {
        return dslContextProvider.ppc(shard)
            .select(CAMPAIGNS.CID)
            .from(CAMPAIGNS)
            .where(CAMPAIGNS.CLIENT_ID.`in`(clientIds))
            .fetch(CAMPAIGNS.CID)
    }

    fun getTyped(
        shard: Int,
        campaignIds: Collection<Long>,
        conditionFilters: List<ConditionFilter> = listOf()
    ): List<BaseCampaign> {
        val queryFilter = createQueryFilter(campaignIds, conditionFilters)
        val resultCampaigns = campaignTypedRepository
            .getTypedCampaigns(
                dslContextProvider.ppc(shard),
                queryFilter
            )
        if (campaignIds.size != resultCampaigns.size) {
            logger.warn("Campaigns ${campaignIds.minus(resultCampaigns.map { it.id })} was filtered")
        }
        return resultCampaigns
    }

    fun getAdGroupIdsWithUnsupportedCampaign(shard: Int, adGroupIds: Collection<Long>): List<Long> {
        return dslContextProvider
            .ppc(shard)
            .select(PHRASES.PID)
            .from(PHRASES)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
            .where(
                CAMPAIGNS.TYPE.`in`(DISALLOWED_CAMPAIGN_DB_TYPES)
                    .and(PHRASES.PID.`in`(adGroupIds))
            )
            .fetch(PHRASES.PID)
    }

    companion object {
        private val logger = LoggerFactory.getLogger(SupportedCampaignsService::class.java)
        private val DISALLOWED_CAMPAIGN_DB_TYPES = run {
            val allowed = CampaignRepositoryConstants.CAMPAIGN_CLASS_TO_JOOQ_TYPE.values
            val all = CampaignsType.values().asList()
            all - allowed
        }

        private object CampaignTypeIsAllowedFilter : ConditionFilter() {
            override fun getCondition(): Condition = CAMPAIGNS.TYPE.notIn(DISALLOWED_CAMPAIGN_DB_TYPES)
            override fun isEmpty(): Boolean = false
        }

        private fun createQueryFilter(
            campaignIds: Collection<Long>,
            conditionFilters: List<ConditionFilter> = listOf()
        ): Filter {
            return ConditionFilterFactory.multipleConditionFilter(
                ConditionFilterFactory.whereInFilter(CAMPAIGNS.CID, campaignIds),
                CampaignTypeIsAllowedFilter,
                *conditionFilters.toTypedArray()
            )
        }
    }
}
