package ru.yandex.direct.logicprocessor.processors.campaignlastchange;


import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.repository.CampAggregatedLastchangeRepository;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.campaignlastchange.CampAggregatedLastchangeConfig;
import ru.yandex.direct.ess.logicobjects.campaignlastchange.CampAggregatedLastchangeObject;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.common.db.PpcPropertyNames.campAggregatedLastchangeHighBoundary;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.CHAT_API_MONITORING;

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        notifications = {
                @OnChangeNotification(recipient = CHAT_API_MONITORING,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        tags = {DIRECT_PRIORITY_0, GROUP_INTERNAL_SYSTEMS, DIRECT_API_TEAM}
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_0, GROUP_INTERNAL_SYSTEMS}
)
@EssLogicProcessor(CampAggregatedLastchangeConfig.class)
public class CampAggregatedLastchangeProcessor extends BaseLogicProcessor<CampAggregatedLastchangeObject> {

    private final CampAggregatedLastchangeService campAggregatedLastchangeService;
    private final CampAggregatedLastchangeRepository campAggregatedLastchangeRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ZoneId defaultZoneId;
    private PpcProperty<LocalDateTime> campAggregatedLastChangeHighBoundary;

    @Autowired
    public CampAggregatedLastchangeProcessor(EssLogicProcessorContext essLogicProcessorContext,
                                             CampAggregatedLastchangeService campAggregatedLastchangeService,
                                             CampAggregatedLastchangeRepository campAggregatedLastchangeRepository,
                                             ZoneId defaultZoneId,
                                             PpcPropertiesSupport ppcPropertiesSupport) {
        super(essLogicProcessorContext);
        this.campAggregatedLastchangeService = campAggregatedLastchangeService;
        this.campAggregatedLastchangeRepository = campAggregatedLastchangeRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.defaultZoneId = defaultZoneId;
    }

    @Override
    protected void initialize() {
        this.campAggregatedLastChangeHighBoundary =
                ppcPropertiesSupport.get(campAggregatedLastchangeHighBoundary(getShard()));
    }

    @Override
    public void process(List<CampAggregatedLastchangeObject> objects) {
        int shard = getShard();
        Map<Long, LocalDateTime> lastChangesByCids =
                campAggregatedLastchangeService.getLastChangesByCids(shard, objects);
        campAggregatedLastchangeRepository.updateCampAggregatedLastchange(shard, lastChangesByCids);
        updatePpcProperty(lastChangesByCids);
        sendMaxLastChangeToSolomon(lastChangesByCids);
    }

    private void updatePpcProperty(Map<Long, LocalDateTime> lastChangesByCids) {
        LocalDateTime minLastChange = lastChangesByCids.values().stream()
                .min(LocalDateTime::compareTo)
                .orElse(null);
        if (Objects.nonNull(minLastChange)) {
            minLastChange = minLastChange.minus(1, ChronoUnit.MINUTES);
            LocalDateTime prevValue = campAggregatedLastChangeHighBoundary.get();
            if (Objects.isNull(prevValue) || prevValue.compareTo(minLastChange) < 0) {
                campAggregatedLastChangeHighBoundary.set(minLastChange);
            }
        }
    }

    private void sendMaxLastChangeToSolomon(Map<Long, LocalDateTime> lastChangesByCids) {
        LocalDateTime maxLastChange = lastChangesByCids.values().stream()
                .max(LocalDateTime::compareTo)
                .orElse(null);
        if (Objects.nonNull(maxLastChange)) {
            getMetricRegistry()
                    .gaugeInt64("max_last_change", Labels.of("host", "CLUSTER"))
                    .set(maxLastChange.atZone(defaultZoneId).toEpochSecond());
        }
    }
}
