package ru.yandex.direct.logicprocessor.processors.campaignlastchange;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.ess.common.utils.TablesEnum;
import ru.yandex.direct.ess.logicobjects.campaignlastchange.CampAggregatedLastchangeObject;

import static ru.yandex.direct.utils.DateTimeUtils.MSK;

@Service
public class CampAggregatedLastchangeService {

    private final BannerRelationsRepository bannerRelationsRepository;
    private final BannerImageRepository bannerImageRepository;
    private final PpcProperty<Boolean> fetchDataFromGrutProp;

    @Autowired
    public CampAggregatedLastchangeService(BannerRelationsRepository bannerRelationsRepository,
                                           BannerImageRepository bannerImageRepository,
                                           PpcPropertiesSupport ppcPropertiesSupport) {
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.bannerImageRepository = bannerImageRepository;
        this.fetchDataFromGrutProp = ppcPropertiesSupport.get(PpcPropertyNames.FETCH_DATA_FROM_GRUT_FOR_CAMP_AGGREGATED_LASTCHANGE,
                Duration.ofMinutes(5));
    }

    Map<Long, LocalDateTime> getLastChangesByCids(int shard, List<CampAggregatedLastchangeObject> objects) {
        List<Long> bannerImagesBids = objects.stream()
                .filter(x -> x.getChangedTable().equals(TablesEnum.BANNER_IMAGES))
                .filter(x -> Objects.nonNull(x.getId()))
                .map(CampAggregatedLastchangeObject::getId)
                .collect(Collectors.toList());

        List<Long> bannerImagesImageIds = objects.stream()
                .filter(x -> x.getChangedTable().equals(TablesEnum.BANNER_IMAGES))
                .filter(x -> Objects.isNull(x.getId()))
                .map(CampAggregatedLastchangeObject::getPrimaryKey)
                .collect(Collectors.toList());

        Map<Long, Long> campaignIdsByBannerIds = bannerImagesBids.isEmpty() ? new HashMap<>() :
                bannerRelationsRepository.getCampaignIdsByBannerIdsForShard(shard, bannerImagesBids);

        Map<Long, Long> campaignIdsByImageIds = bannerImagesImageIds.isEmpty() ? new HashMap<>() :
                bannerImageRepository.getCampaignIdsByImageIdsForShard(shard, bannerImagesImageIds);

        if (!fetchDataFromGrutProp.getOrDefault(false)) {
            objects = StreamEx.of(objects)
                    .filter(object -> object.getChangedTable() != TablesEnum.BANNER_CANDIDATES)
                    .toList();
        }

        return objects.stream()
                .map(object -> Pair
                        .of(getCidById(object, campaignIdsByBannerIds, campaignIdsByImageIds), object.getLastChange()))
                .filter(pair -> Objects.nonNull(pair.getLeft()))
                .filter(pair -> Objects.nonNull(pair.getRight()))
                .collect(Collectors.toMap(Pair::getLeft, pair -> pair.getRight().atZone(MSK).toLocalDateTime(),
                        (lastChange1, lastChange2) -> lastChange1.compareTo(lastChange2) > 0 ? lastChange1
                                : lastChange2));


    }

    private Long getCidById(CampAggregatedLastchangeObject object, Map<Long, Long> campaignIdsByBannerIds,
                            Map<Long, Long> campaignIdsByImageIds) {
        switch (object.getChangedTable()) {
            case BANNERS:
            case PHRASES:
            case BANNER_CANDIDATES:
                return object.getId();
            case BANNER_IMAGES:
                return campaignIdsByBannerIds.get(object.getId()) != null ? campaignIdsByBannerIds.get(object.getId())
                        : campaignIdsByImageIds.get(object.getPrimaryKey());
            default:
                throw new IllegalStateException("unsupported table");
        }
    }
}
