package ru.yandex.direct.logicprocessor.processors.campaignstatuscorrect;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithPricePackage;
import ru.yandex.direct.core.entity.banner.model.BannerWithStatusActive;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightReasonIncorrect;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusCorrect;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.ess.logicobjects.campaignstatuscorrect.CampaignStatusCorrectCheckObject;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.isCampaignFullWithBanners;
import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class CampaignStatusCorrectCheckService {

    private final DslContextProvider dslContextProvider;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final CreativeRepository creativeRepository;
    private final PriceCampaignsMinusGeoCheckService priceCampaignsMinusGeoCheckService;

    @Autowired
    public CampaignStatusCorrectCheckService(
            DslContextProvider dslContextProvider,
            CampaignTypedRepository campaignTypedRepository,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            BannerTypedRepository bannerTypedRepository,
            CreativeRepository creativeRepository,
            PriceCampaignsMinusGeoCheckService priceCampaignsMinusGeoCheckService) {
        this.dslContextProvider = dslContextProvider;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.creativeRepository = creativeRepository;
        this.priceCampaignsMinusGeoCheckService = priceCampaignsMinusGeoCheckService;
    }

    public List<Long> getCpmPriceCampaigns(int shard, List<CampaignStatusCorrectCheckObject> objects) {
        var campaignIds = listToSet(objects, o -> checkNotNull(o.getCid()));
        var campaignIdToType = campaignRepository.getCampaignsTypeMap(shard, campaignIds);
        return EntryStream.of(campaignIdToType)
                .filterValues(type -> type == CampaignType.CPM_PRICE)
                .keys()
                .toList();
    }

    /**
     * 1) Дефолтная группа существует и промодерирована
     * 2) Существует ровно одна дефолтная группа
     * 3) В дефолтной группе есть промодерированный активный баннер (не остановлен, не заархивирован, промодерирован)
     * с соответствующим форматом креатива (тоже должен быть промодерированным) под каждую версию морды
     *
     * @param campaignIds коллекция идентификаторов кампаний
     */
    public void verifyCampaignsStatusCorrect(int shard, Collection<Long> campaignIds) {
        var cpmPriceCampaigns = (List<CpmPriceCampaign>) campaignTypedRepository.getTypedCampaigns(shard, campaignIds);
        var campaignIdToAdGroupIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds);
        var adgroupTypes = adGroupRepository.getAdGroupTypesByIds(shard, flatToList(campaignIdToAdGroupIds.values()));
        var campaignIdCpmYndxOnly = new HashMap<Long, Boolean>();
        campaignIdToAdGroupIds.forEach((key, val) -> {
            var adGroupTypes = filterToSet(mapList(campaignIdToAdGroupIds.get(key), adgroupTypes::get),
                    Objects::nonNull);
            campaignIdCpmYndxOnly.put(key,
                    adGroupTypes.size() == 1 && adGroupTypes.contains(AdGroupType.CPM_YNDX_FRONTPAGE)
                            || adGroupTypes.isEmpty());
        });

        dslContextProvider.ppcTransaction(shard, conf -> {
            List<Long> campaignIdsToResend = new ArrayList<>();
            StreamEx.of(cpmPriceCampaigns)
                    .mapToEntry(campaign -> campaignIdCpmYndxOnly.getOrDefault(campaign.getId(), true) ?
                            isYndxFrontpageCampaignStatusCorrect(shard, campaign,
                                    campaignIdToAdGroupIds.getOrDefault(campaign.getId(), emptyList())) :
                            isOtherCampaignStatusCorrect(shard, campaign, campaignIdToAdGroupIds.get(campaign.getId())))
                    .forKeyValue((campaign, reasonIncorrect) -> {
                        var dbStatusCorrect = campaign.getFlightStatusCorrect();
                        var statusCorrect = reasonIncorrect == null ? PriceFlightStatusCorrect.YES :
                                PriceFlightStatusCorrect.NO;
                        campaignRepository.setCampaignsCpmPriceStatusCorrect(conf, campaign.getId(), statusCorrect,
                                reasonIncorrect);

                        //если статус изменился, то переотправляем кампанию в БК
                        if (dbStatusCorrect != statusCorrect) {
                            campaignIdsToResend.add(campaign.getId());
                        }
                    });
            campaignRepository.updateStatusBsSynced(conf.dsl(), campaignIdsToResend, StatusBsSynced.NO);
        });
    }

    @Nullable
    private PriceFlightReasonIncorrect isYndxFrontpageCampaignStatusCorrect(int shard, CpmPriceCampaign campaign,
                                                                            List<Long> adGroupIds) {
        var adGroupIdToPriority = adGroupRepository.getAdGroupsPriority(shard, adGroupIds);
        var defaultAdGroupIds = EntryStream.of(adGroupIdToPriority)
                .filterValues(AdGroupCpmPriceUtils::isDefaultPriority)
                .keys()
                .toList();

        var reasonIncorrect = verifyDefaultGroupsSize(defaultAdGroupIds);
        if (reasonIncorrect != null) {
            return reasonIncorrect;
        }

        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, defaultAdGroupIds);
        if (adGroups.isEmpty()) {
            return PriceFlightReasonIncorrect.NO_DEFAULT_GROUP;
        }

        AdGroup defaultAdGroup = adGroups.get(0);
        return isAdGroupFull(shard, campaign, defaultAdGroup);
    }

    @Nullable
    private PriceFlightReasonIncorrect isOtherCampaignStatusCorrect(int shard, CpmPriceCampaign campaign,
                                                                    List<Long> adGroupIdsForCheck) {

        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, adGroupIdsForCheck);
        if (adGroups.isEmpty()) {
            return PriceFlightReasonIncorrect.NOT_FULL;
        }

        PriceFlightReasonIncorrect adGroupCheckResult = null;
        for (AdGroup adGroup : adGroups) {
            adGroupCheckResult = isAdGroupFull(shard, campaign, adGroup);
            if (adGroupCheckResult == null) {
                //Если хотя бы одна группа нормальная - то считаем кампанию корректной.
                return null;
            }
        }
        // Если так и не нашлось ни одной группы хорошей - возвращаем последнюю ошибку найденную.
        return adGroupCheckResult;
    }

    @Nullable
    private PriceFlightReasonIncorrect verifyDefaultGroupsSize(List<Long> defaultAdGroupIds) {
        if (defaultAdGroupIds.isEmpty()) {
            return PriceFlightReasonIncorrect.NO_DEFAULT_GROUP;
        } else if (defaultAdGroupIds.size() > 1) {
            return PriceFlightReasonIncorrect.MORE_THAN_ONE_DEFAULT_GROUP;
        }
        return null;
    }

    private PriceFlightReasonIncorrect isAdGroupFull(int shard, CpmPriceCampaign campaign,
                                                     AdGroup adGroup) {
        List<BannerWithSystemFields> banners = mapList(bannerTypedRepository
                        .getBannersByGroupIds(shard, singletonList(adGroup.getId())),
                b -> (BannerWithSystemFields) b);

        if ((adGroup.getStatusModerate() != ru.yandex.direct.core.entity.adgroup.model.StatusModerate.YES
                || adGroup.getStatusPostModerate() != ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate.YES)
                && banners.stream().noneMatch(BannerWithStatusActive::getStatusActive)) {
            return (adGroup.getType() == AdGroupType.CPM_YNDX_FRONTPAGE) ?
                    PriceFlightReasonIncorrect.NOT_MODERATED_DEFAULT_GROUP :
                    PriceFlightReasonIncorrect.NOT_MODERATED_GROUP;
        }

        if (isEmpty(banners)) {
            return PriceFlightReasonIncorrect.NO_ACTIVE_BANNERS;
        }

        var activeBanners = StreamEx.of(banners)
                .select(BannerWithPricePackage.class)
                .filter(CampaignWithPricePackageUtils::isBannerActive)
                .toList();

        if (isEmpty(activeBanners)) {
            return PriceFlightReasonIncorrect.NO_ACTIVE_BANNERS;
        }

        var bannersWithValidGeo =
                priceCampaignsMinusGeoCheckService.filterBannersWithoutOverlappingMinusRegions(shard, activeBanners);

        if (bannersWithValidGeo.size() != activeBanners.size()) {
            return PriceFlightReasonIncorrect.NOT_MODERATED_DEFAULT_GROUP;
        }

        if (adGroup.getType() == AdGroupType.CPM_YNDX_FRONTPAGE) {
            var bannerIdToCreatives =
                    creativeRepository.getCreativesByBannerIds(shard, mapList(activeBanners, Banner::getId));
            if (!isCampaignFullWithBanners(campaign, activeBanners, bannerIdToCreatives)) {
                return PriceFlightReasonIncorrect.NOT_FULL;
            }
        }

        return null;
    }
}
