package ru.yandex.direct.logicprocessor.processors.campaignstatuscorrect;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithPricePackage;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.regions.GeoTree;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultPriority;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
public class PriceCampaignsMinusGeoCheckService {

    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final PricePackageService pricePackageService;
    private final DslContextProvider dslContextProvider;
    private final BannerModerationRepository bannerModerationRepository;

    @Autowired
    public PriceCampaignsMinusGeoCheckService(
            CampaignRepository campaignRepository,
            CampaignTypedRepository campaignTypedRepository,
            AdGroupRepository adGroupRepository,
            PricePackageService pricePackageService,
            DslContextProvider dslContextProvider, BannerModerationRepository bannerModerationRepository) {
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.pricePackageService = pricePackageService;
        this.dslContextProvider = dslContextProvider;
        this.bannerModerationRepository = bannerModerationRepository;
    }

    /**
     * filterBannersWithoutOverlappingMinusRegions(shard, banners)
     * <p>
     * В переданных баннерах удаляет баннеры, которые принадлежат дефолтной группе и имеют минус-регионы
     * пересекающиеся с гео кампании и возвращают получившийся список
     *
     * @param shard
     * @param banners
     * @return List<BannerWithPricePackage>
     */

    public List<BannerWithPricePackage> filterBannersWithoutOverlappingMinusRegions(int shard,
                                                                                    List<BannerWithPricePackage> banners) {
        DSLContext ctx = dslContextProvider.ppc(shard);
        Set<Long> bannerIds = filterAndMapToSet(banners, e -> true, BannerWithPricePackage::getId);

        Map<Long, BannerWithPricePackage> bannersMap = listToMap(banners, BannerWithPricePackage::getId);

        Set<Long> campaignIds = listToSet(banners, BannerWithPricePackage::getCampaignId);
        Map<Long, CampaignType> campaignTypeMap = campaignRepository.getCampaignsTypeMap(ctx, campaignIds);

        Set<Long> cpmPriceAdGroupIds = listToSet(banners, BannerWithPricePackage::getAdGroupId);
        Map<Long, Long> cpmPriceAdGroupPriorities = adGroupRepository.getAdGroupsPriority(ctx, cpmPriceAdGroupIds);
        removeNotDefaultAdgroups(bannerIds, bannersMap, cpmPriceAdGroupPriorities);

        Set<Long> reachableCampaignIds = bannerIds.stream()
                .map(bannersMap::get)
                .map(BannerWithPricePackage::getCampaignId)
                .collect(Collectors.toSet());
        removeUnreachableCampaigns(campaignTypeMap, reachableCampaignIds);

        Map<Long, CpmPriceCampaign> cpmPriceCampaigns = campaignTypedRepository.getTypedCampaignsMap(
                ctx, campaignTypeMap, CampaignType.CPM_PRICE);

        var bannersMinusGeo = bannerModerationRepository.getBannersMinusGeo(shard, bannerIds);

        return banners.stream()
                .filter(banner -> !bannerMinusRegionsOverlapWithCampaign(bannersMinusGeo.getOrDefault(banner.getId(),
                        emptyList()),
                        cpmPriceCampaigns.get(banner.getCampaignId())))
                .collect(Collectors.toList());

    }

    private void removeNotDefaultAdgroups(Collection<Long> bannerIds,
                                          Map<Long, BannerWithPricePackage> bannersMap,
                                          Map<Long, Long> cpmPriceAdGroupPriorities) {
        bannerIds.removeIf(bannerId -> {
            var adGroupId = bannersMap.get(bannerId).getAdGroupId();
            return !isDefaultPriority(cpmPriceAdGroupPriorities.get(adGroupId));
        });
    }

    private void removeUnreachableCampaigns(Map<Long, CampaignType> campaignTypeMap,
                                            Set<Long> reachableCampaignIds) {
        campaignTypeMap.entrySet().removeIf(e -> !reachableCampaignIds.contains(e.getKey()));
    }

    private boolean bannerMinusRegionsOverlapWithCampaign(List<Long> minusRegions,
                                                          CpmPriceCampaign cpmPriceCampaign) {

        if (isEmpty(minusRegions)) {
            // geoTree.isAnyRegionOrSubRegionIncludedIn возвращает true на пустой minusRegionsSet
            // нам этом не подходит, поэтому делаем здесь return false
            return false;
        }

        Set<Long> campaignRegionsSet = new HashSet<>(cpmPriceCampaign.getFlightTargetingsSnapshot().getGeoExpanded());
        Set<Long> minusRegionsSet = new HashSet<>(minusRegions);
        return getGeoTree().isAnyRegionOrSubRegionIncludedIn(minusRegionsSet, campaignRegionsSet);
    }

    private GeoTree getGeoTree() {
        return pricePackageService.getGeoTree();
    }
}
