package ru.yandex.direct.logicprocessor.processors.campeventlog;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.eventlog.model.EventLog;
import ru.yandex.direct.core.entity.eventlog.model.EventLogParams;
import ru.yandex.direct.core.entity.eventlog.model.EventLogType;
import ru.yandex.direct.core.entity.eventlog.repository.EventLogRepository;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.campeventlog.CampaignEventLogConfig;
import ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject;
import ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject.BudgetType;
import ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject.CampaignEventType;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;

import static ru.yandex.direct.core.entity.eventlog.model.EventLogType.AVG_CPA_CHANGED;
import static ru.yandex.direct.core.entity.eventlog.model.EventLogType.DAILY_BUDGET_CHANGED;
import static ru.yandex.direct.core.entity.eventlog.model.EventLogType.ROI_COEF_CHANGED;
import static ru.yandex.direct.core.entity.eventlog.model.EventLogType.WEEKLY_BUDGET_CHANGED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Процессор для записи изменений таблицы ppc.campaigns в ppc.eventlog. В данный момент логируются только изменения
 * недельного (campaigns.strategy_data.sum) и дневного (campaigns.day_budget) бюджетов.
 */
@ParametersAreNonnullByDefault
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 5), needCheck = ProductionOnly.class,
        //PRIORITY: Временно поставили приоритет по умолчанию; gerdler
        tags = {DIRECT_PRIORITY_1})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class,
        //PRIORITY: Временно поставили приоритет по умолчанию; gerdler
        tags = {DIRECT_PRIORITY_1})
@EssLogicProcessor(CampaignEventLogConfig.class)
public class CampaignEventLogProcessor extends BaseLogicProcessor<CampaignEventLogLogicObject> {

    private final EventLogRepository eventLogRepository;

    public CampaignEventLogProcessor(EssLogicProcessorContext essLogicProcessorContext,
                                     EventLogRepository eventLogRepository) {
        super(essLogicProcessorContext);

        this.eventLogRepository = eventLogRepository;
    }

    @Override
    public void process(List<CampaignEventLogLogicObject> logicObjects) {
        List<EventLog> eventLogs = StreamEx.of(logicObjects)
                .map(logicObject ->
                                new EventLog()
                                        .withCampaignId(logicObject.getCampaignId())
                                        .withClientId(logicObject.getClientId().asLong())
                                 //todo вернуть после удаления перечисления BudgetType и удалить следующую конвертацию
//                                      .withType(convertBudgetTypeToEventLogType(logicObject.getCampaignEventType()))
                                        .withType(logicObject.getBudgetType() != null
                                                ? convertBudgetTypeToEventLogType(logicObject.getBudgetType()) :
                                                (logicObject.getCampaignEventType() != null
                                                        ?
                                                        convertBudgetTypeToEventLogType(
                                                                logicObject.getCampaignEventType()) : null))
                                        .withBidsId(0L)
                                        .withBannerId(0L)
                                        .withEventTime(logicObject.getEventTime())
                                        .withParams(
                                                new EventLogParams()
                                                        .withBudgetBefore(logicObject.getBudgetBefore())
                                                        .withBudgetAfter(logicObject.getBudgetAfter())
                                                        .withAvgCpaBefore(logicObject.getAvgCpaBefore())
                                                        .withAvgCpaAfter(logicObject.getAvgCpaAfter())
                                                        .withRoiCoefBefore(logicObject.getRoiCoefBefore())
                                                        .withRoiCoefAfter(logicObject.getRoiCoefAfter())
                                        )
                )
                .toList();

        eventLogRepository.addEventLogs(getShard(), eventLogs);
    }

    private static EventLogType convertBudgetTypeToEventLogType(BudgetType budgetType) {
        switch (budgetType) {
            case DAILY:
                return DAILY_BUDGET_CHANGED;
            case WEEKLY:
                return WEEKLY_BUDGET_CHANGED;
        }

        throw new IllegalArgumentException("Unknown budget type: " + budgetType);
    }

    private static EventLogType convertBudgetTypeToEventLogType(CampaignEventType campaignEventType) {
        switch (campaignEventType) {
            case DAILY_BUDGET_CHANGED:
                return DAILY_BUDGET_CHANGED;
            case WEEKLY_BUDGET_CHANGED:
                return WEEKLY_BUDGET_CHANGED;
            case AVG_CPA_CHANGED:
                return AVG_CPA_CHANGED;
            case ROI_COEF_CHANGED:
                return ROI_COEF_CHANGED;
        }

        throw new IllegalArgumentException("Unknown campaign event type: " + campaignEventType);
    }
}
