package ru.yandex.direct.logicprocessor.processors.campstatusmoderate;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesCampaignRepository;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForBlockedMoneyCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.direct.ess.logicobjects.moderation.campstatusmoderate.CampaignStatusEventsObject;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.intapi.client.model.response.CampStatusModerate;
import ru.yandex.direct.logicprocessor.processors.campstatusmoderate.handlers.BalanceResyncHandler;
import ru.yandex.direct.logicprocessor.processors.campstatusmoderate.handlers.CampaignStatusHandler;
import ru.yandex.direct.logicprocessor.processors.campstatusmoderate.handlers.CampaignToResync;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_CRIT;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_PROCESSING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_WARN;
import static ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate.ACCEPTED;
import static ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate.NO;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Service
public class CampaignStatusCalcService {
    private static final Logger logger = LoggerFactory.getLogger(CampaignStatusCalcService.class);

    private final AggregatedStatusesCampaignRepository aggregatedStatusesCampaignRepository;
    private final CampaignRepository campaignRepository;

    private final CampaignService campaignService;
    private final List<CampaignStatusHandler> handlers;
    private final PpcProperty<Integer> clientIdPercentProperty;
    private final PpcProperty<Integer> newCalculatorPercentProperty;
    private final Set<Long> ignoredClientIds;

    private final BalanceResyncHandler balanceResyncHandler;
    private final IntApiClient intApiClient;

    @Autowired
    public CampaignStatusCalcService(AggregatedStatusesCampaignRepository aggregatedStatusesCampaignRepository,
                                     CampaignRepository campaignRepository,
                                     List<CampaignStatusHandler> campaignStatusHandlers,
                                     CampaignService campaignService, PpcPropertiesSupport ppcPropertiesSupport,
                                     DirectConfig directConfig,
                                     BalanceResyncHandler balanceResyncHandler,
                                     IntApiClient intApiClient) {
        this.aggregatedStatusesCampaignRepository = aggregatedStatusesCampaignRepository;
        this.campaignRepository = campaignRepository;
        this.campaignService = campaignService;
        this.handlers = List.copyOf(campaignStatusHandlers);
        this.balanceResyncHandler = balanceResyncHandler;
        this.intApiClient = intApiClient;
        this.clientIdPercentProperty = ppcPropertiesSupport.get(PpcPropertyNames.AGGREGATED_CAMP_STATUS_MODERATE_JAVA,
                Duration.ofMinutes(1));
        this.newCalculatorPercentProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.AGGREGATED_CAMP_STATUS_MODERATE_JAVA_PREFER_NEW_CALCULATOR,
                        Duration.ofMinutes(1));

        if (directConfig.hasPath("ignore_test_clients_in_transport")) {
            this.ignoredClientIds = Set.copyOf(directConfig.getLongList("ignore_test_clients_in_transport"));
        } else {
            this.ignoredClientIds = Set.of();
        }
    }

    private Map<Long, Boolean> isMoneyBlocked(List<AggregatedStatusCampaign> cids) {
        List<CampaignForBlockedMoneyCheck> campaigns = mapList(cids,
                row -> new Campaign()
                        .withId(row.getId())
                        .withType(row.getType())
                        .withWalletId(row.getWalletId())
                        .withUserId(row.getUserId())
                        .withManagerUserId(row.getManagerUid())
                        .withAgencyUserId(row.getAgencyUid())
                        .withStatusPostModerate(row.getStatusPostModerate())
                        .withStatusModerate(row.getStatusModerate())
                        .withSum(row.getSum())
                        .withSumToPay(row.getSumToPay())
        );

        return campaignService.moneyOnCampaignsIsBlocked(campaigns, false, false, false);
    }

    /*
    my $status_post_moderate = $current_camp->{statusPostModerate} eq 'Accepted' || $O{StatusModerate} eq
     'Yes' ?
278                                 'Accepted'
279                                 : (check_block_money_camp($current_camp) ? undef : 'No');
 */
    @Nullable
    CampaignStatusPostmoderate calcStatusPostModerate(AggregatedStatusCampaign campaign,
                                                      @NotNull CampaignStatusModerate newStatusModerate,
                                                      boolean moneyBlocked) {
        if (ACCEPTED == campaign.getStatusPostModerate() || newStatusModerate == CampaignStatusModerate.YES) {
            return ACCEPTED;
        } else {
            if (moneyBlocked) {
                return null;
            } else {
                return NO;
            }
        }
    }

    /*
170     my $new_statusBsSynced;
171     if (! $current_camp->{statusPostModerate}
172         || (
173             $new_status_post_moderate
174             && $current_camp->{statusPostModerate} eq $new_status_post_moderate
175             # закрываем случай гонок с экспортом в БК, когда Yes на кампании ставится с большим опозданием от
баннеров
176             && ($current_camp->{statusModerate} eq 'Yes' || $Status ne 'Yes' || $current_camp->{OrderID} )
177             )
178     ) {
179         # empty
180     } else {
181         # переотправляем в БК, если были изменения в статусе
182         $new_statusBsSynced = 'No';
183     }
     */
    boolean needResyncWithBS(AggregatedStatusCampaign campaign,
                             CampaignStatusModerate newStatusModerate,
                             @Nullable CampaignStatusPostmoderate newStatusPostModerate) {
        if (campaign.getStatusPostModerate() == null) {
            // Колонка not null, поэтому если здесь null, то это означает, что у кампании нет записи в camp_options
            return false;
        }
        return newStatusPostModerate == null || campaign.getStatusPostModerate() != newStatusPostModerate
                || campaign.getStatusModerate() != CampaignStatusModerate.YES
                && newStatusModerate == CampaignStatusModerate.YES
                // DIRECT-55693
                && nvl(campaign.getOrderId(), 0L) <= 0;
    }

    private boolean isInternalCampaign(CampaignType campaignType) {
        return CampaignTypeKinds.INTERNAL.contains(campaignType);
    }

    private boolean isPerformanceCampaign(CampaignType campaignType) {
        return campaignType == CampaignType.PERFORMANCE;
    }

    boolean isIgnoredCampaignType(CampaignType campaignType) {
        return isInternalCampaign(campaignType) || isPerformanceCampaign(campaignType);
    }

    public void process(int shard, List<CampaignStatusEventsObject> logicObjects) {
        List<Long> cids = mapList(logicObjects, CampaignStatusEventsObject::getCampaignId);

        int percent = clientIdPercentProperty.getOrDefault(0);
        List<AggregatedStatusCampaign> allCampaigns = aggregatedStatusesCampaignRepository.getCampaigns(shard, cids);
        List<AggregatedStatusCampaign> campaigns = allCampaigns.stream()
                .filter(c -> c.getClientId() % 100 < percent)
                // Для кампаний внутренней рекламы статусы модерации/постмодерации не пересчитываются:
                // после отправки на модерацию такие кампании сразу получают Yes/Accepted
                // и после этого уже не изменяются
                // performance-кампании работают аналогично
                .filter(c -> !isIgnoredCampaignType(c.getType()))
                .collect(toList());
        if (campaigns.isEmpty()) {
            return;
        }

        logger.info("Processing cids: {}", mapList(campaigns, AggregatedStatusCampaign::getId));

        Map<Long, Boolean> moneyBlockedMap = isMoneyBlocked(campaigns);

        Map<Long, CampStatusModerate> perlComputedStatuses;
        // Если включено на 100%, то не делаем сравнение с perl
        // Если же понадобится вновь включить сравнение с perl, можно попробовать поставить 101%
        boolean ignorePerlImplementation = (percent == 100);
        if (ignorePerlImplementation) {
            perlComputedStatuses = emptyMap();
        } else {
            perlComputedStatuses = intApiClient.calculateCampaignStatusModerateReadOnly(mapList(campaigns,
                    AggregatedStatusCampaign::getId));
        }

        Map<CampaignStatus, List<Long>> statusesToCids = new HashMap<>();
        List<AggregatedStatusCampaign> campaignsMoneyUnblocked = new ArrayList<>();
        for (AggregatedStatusCampaign campaign : campaigns) {
            if (ignoredClientIds.contains(campaign.getClientId())) {
                logger.info("Skip ignored client: {}", campaign.getClientId());
                continue;
            }

            boolean moneyBlocked = moneyBlockedMap.getOrDefault(campaign.getId(), false);

            CampaignStatus state;
            if (ignorePerlImplementation) {
                state = processOneCampaign(campaign, moneyBlocked);
            } else {
                state = processOneCampaign(campaign, moneyBlocked, perlComputedStatuses.get(campaign.getId()));
            }
            if (state == null) {
                continue;
            }

            statusesToCids.computeIfAbsent(state, e -> new ArrayList<>()).add(campaign.getId());

            if (campaign.getStatusPostModerate() != ACCEPTED && state.getStatusPostModerate() == ACCEPTED && moneyBlocked) {
                campaignsMoneyUnblocked.add(campaign);
            }
        }

        if (statusesToCids.isEmpty()) {
            return;
        }

        // Применяем изменения в базе
        for (CampaignStatusHandler handler : handlers) {
            handler.process(shard, statusesToCids);
        }
        // Логируем изменённые кампании, чтобы в случае чего можно было грепнуть их из логов
        logger.info("Changed campaigns: {}",
                statusesToCids.values().stream().flatMap(Collection::stream).distinct().collect(toList()));

        // Кампании с "заблокированными" деньгами, которые теперь в силу изменения статуса модерации
        // "разблокировались", нужно переотправить в Баланс, чтобы туда отправился флаг unmoderated=0
        // Если этого не сделать, то через некоторое время произойдёт автоматический возврат средств DIRECT-15514
        // "Заблокированные" деньги -- это деньги на кампании, которая не прошла модерацию и потому не может крутиться
        // Стоимость возврата их клиенту часто ненулевая, и поэтому мы стараемся сделать так, чтобы возврата избежать
        if (!campaignsMoneyUnblocked.isEmpty()) {
            List<CampaignToResync> campsSendToBalance = getAllCampaignsWithSameWallet(shard, campaignsMoneyUnblocked);
            if (!campsSendToBalance.isEmpty()) {
                balanceResyncHandler.addToBalanceInfoQueue(shard, campsSendToBalance);
            }
        }
    }

    List<CampaignToResync> getAllCampaignsWithSameWallet(int shard, List<AggregatedStatusCampaign> campaigns) {
        Preconditions.checkState(!campaigns.isEmpty());

        var walletIds = StreamEx.of(campaigns).map(AggregatedStatusCampaign::getWalletId).nonNull().toSet();
        var walletsWithCampaigns = campaignRepository.getWalletsWithCampaignsByWalletCampaignIds(shard, walletIds,
                false);
        var campaignsUnderWallets = StreamEx.of(walletsWithCampaigns.getAllWallets())
                .map(walletsWithCampaigns::getCampaignsBoundTo)
                .flatMap(Collection::stream)
                .map(c -> new CampaignToResync(c.getId(), c.getUserId()))
                .toSet();

        Set<CampaignToResync> result = new HashSet<>(campaignsUnderWallets);
        // Добавляем и исходные кампании, т.к. там могли оказаться кампании без общего счёта
        result.addAll(mapList(campaigns, c -> new CampaignToResync(c.getId(), c.getUserId())));

        return StreamEx.of(result).toList();
    }

    @Nullable
    CampaignStatusModerate calcStatusModerate(Map<GdSelfStatusEnum, Integer> statusesMap) {
        Set<GdSelfStatusEnum> allRun = GdSelfStatusEnum.allRun();
        Set<GdSelfStatusEnum> allCrit = Set.of(STOP_CRIT, STOP_WARN);
        List<GdSelfStatusEnum> statuses = EntryStream.of(statusesMap).filterValues(v -> v > 0).keys().toList();
        // STOP_PROCESSING добавлен для динамических объявлений:
        // если группа находится в статусе STOP_PROCESSING, это значит, что все баннеры находятся в статусе
        // RUN_PROCESSING (такая особенность агр статусов в ДО), поэтому мы можем проставить statusModerate=Yes
        if (statuses.stream().anyMatch(status -> allRun.contains(status) || status == STOP_PROCESSING)) {
            return CampaignStatusModerate.YES;
        } else if (statuses.stream().anyMatch(allCrit::contains)) {
            return CampaignStatusModerate.NO;
        }
        return null;
    }

    CampaignStatus processOneCampaign(AggregatedStatusCampaign campaign, boolean moneyBlocked,
                                      CampStatusModerate perlCalculatedStatus) {
        CampaignStatus campaignStatus = processOneCampaign(campaign, moneyBlocked);
        return mergeCalculatedStatuses(campaign, campaignStatus, perlCalculatedStatus);
    }

    /**
     * Сравнивает статусы, посчитанные новым кодом и перлом
     * В случае расхождений использует перловый результат
     * Возвращает то, что получилось в результате
     */
    CampaignStatus mergeCalculatedStatuses(AggregatedStatusCampaign campaign,
                                           @Nullable CampaignStatus campaignStatus,
                                           CampStatusModerate perlCalculatedStatus) {
        CampaignStatusModerate perlTargetStatusModerate;
        if (perlCalculatedStatus.getStatusModerate() != null) {
            perlTargetStatusModerate = CampaignStatusModerate.fromSource(
                    CampaignsStatusmoderate.valueOf(perlCalculatedStatus.getStatusModerate()));
        } else {
            perlTargetStatusModerate = campaign.getStatusModerate();
        }
        CampaignStatusPostmoderate perlTargetStatusPostModerate;
        if (perlCalculatedStatus.getStatusPostModerate() != null) {
            perlTargetStatusPostModerate = CampaignStatusPostmoderate.fromSource(
                    CampOptionsStatuspostmoderate.valueOf(perlCalculatedStatus.getStatusPostModerate()));
        } else {
            perlTargetStatusPostModerate = campaign.getStatusPostModerate();
        }
        CampaignStatusModerate targetStatusModerate = firstNonNull(
                campaignStatus != null ? campaignStatus.getStatusModerate() : campaign.getStatusModerate(),
                campaign.getStatusModerate());
        CampaignStatusPostmoderate targetStatusPostModerate = firstNonNull(
                campaignStatus != null ? campaignStatus.getStatusPostModerate() : campaign.getStatusPostModerate(),
                campaign.getStatusPostModerate());

        if (targetStatusModerate == perlTargetStatusModerate
                && targetStatusPostModerate == perlTargetStatusPostModerate) {
            return campaignStatus;
        } else {
            //
            int percent = newCalculatorPercentProperty.getOrDefault(0);
            boolean useNewCalculator = campaign.getClientId() % 100 < percent;
            //
            logger.warn("Target statuses differs for cid {}: new code wants {} but perl says {}." +
                            "Current statuses are {};{}. Aggr status {}. Will use {} result.",
                    campaign.getId(), campaignStatus, perlCalculatedStatus,
                    campaign.getStatusModerate(), campaign.getStatusPostModerate(),
                    toJson(campaign.getAggregatedStatus()),
                    useNewCalculator ? "java" : "perl"
            );
            if (useNewCalculator) {
                return campaignStatus;
            }
            //
            if (perlTargetStatusModerate == campaign.getStatusModerate()
                    && perlTargetStatusPostModerate == campaign.getStatusPostModerate()) {
                return null;
            } else {
                return new CampaignStatus(
                        perlTargetStatusModerate,
                        perlTargetStatusPostModerate,
                        needResyncWithBS(campaign, perlTargetStatusModerate, perlTargetStatusPostModerate)
                );
            }
        }
    }

    CampaignStatus processOneCampaign(AggregatedStatusCampaign campaign, boolean moneyBlocked) {
        if (campaign.getAggregatedStatus() == null || campaign.getAggregatedStatus().getCounters() == null
                || campaign.getAggregatedStatus().getCounters().getStatuses() == null) {
            return null;
        }

        Map<GdSelfStatusEnum, Integer> statuses = campaign.getAggregatedStatus().getCounters().getStatuses();

        CampaignStatusModerate statusModerate = calcStatusModerate(statuses);
        if (statusModerate == null) {
            return null;
        }
        CampaignStatusPostmoderate statusPostModerate = calcStatusPostModerate(campaign, statusModerate, moneyBlocked);
        boolean needResyncWithBs = needResyncWithBS(campaign, statusModerate, statusPostModerate);

        CampaignStatus campaignStatus = new CampaignStatus(statusModerate, statusPostModerate, needResyncWithBs);
        var targetStatusModerate = firstNonNull(campaignStatus.getStatusModerate(), campaign.getStatusModerate());
        var targetStatusPostModerate = firstNonNull(campaignStatus.getStatusPostModerate(),
                campaign.getStatusPostModerate());

        // если желаемый статус не отличается от текущего, то не пытаемся применить эти изменения, возвращая null
        if (targetStatusModerate == campaign.getStatusModerate()
                && targetStatusPostModerate == campaign.getStatusPostModerate()
                && !needResyncWithBs) {
            return null;
        } else {
            return campaignStatus;
        }
    }
}
