package ru.yandex.direct.logicprocessor.processors.campstatusmoderate.handlers;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.jooq.TableField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.BalanceInfoQueueObjType;
import ru.yandex.direct.dbschema.ppc.enums.BalanceInfoQueueSendStatus;
import ru.yandex.direct.dbschema.ppc.tables.records.BalanceInfoQueueRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.BALANCE_INFO_QUEUE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class BalanceResyncHandler {
    private static final Logger logger = LoggerFactory.getLogger(BalanceResyncHandler.class);

    private final DslContextProvider dslContextProvider;

    // Приоритет как в перле
    private static final int PRIORITY_CAMP_WITH_BLOCKED_MONEY = 115;
    private static final int CHUNK_SIZE = 200;

    @Autowired
    public BalanceResyncHandler(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public void addToBalanceInfoQueue(int shard, List<CampaignToResync> campaigns) {
        if (campaigns.isEmpty()) {
            return;
        }
        List<List<CampaignToResync>> chunks = Lists.partition(campaigns, CHUNK_SIZE);
        for (List<CampaignToResync> chunk : chunks) {
            processChunk(shard, chunk);
        }
    }

    private void processChunk(int shard, List<CampaignToResync> campaigns) {
        List<Long> campaignIds = mapList(campaigns, c -> c.campaignId);
        Set<Long> existing = dslContextProvider.ppc(shard)
                .selectDistinct(BALANCE_INFO_QUEUE.CID_OR_UID)
                .from(BALANCE_INFO_QUEUE)
                .where(BALANCE_INFO_QUEUE.SEND_STATUS.eq(BalanceInfoQueueSendStatus.Wait))
                .and(BALANCE_INFO_QUEUE.OBJ_TYPE.eq(BalanceInfoQueueObjType.cid))
                .and(BALANCE_INFO_QUEUE.CID_OR_UID.in(campaignIds))
                .and(BALANCE_INFO_QUEUE.PRIORITY.ge((long) PRIORITY_CAMP_WITH_BLOCKED_MONEY))
                .fetchSet(BALANCE_INFO_QUEUE.CID_OR_UID);
        if (!existing.isEmpty()) {
            logger.info("Skip {} campaigns that are already in the queue: {}", existing.size(), existing);
        }
        List<CampaignToResync> campaignsToInsert = campaigns.stream()
                .filter(campaign -> !existing.contains(campaign.campaignId))
                .collect(Collectors.toList());
        if (!campaignsToInsert.isEmpty()) {
            insertIntoQueue(shard, campaignsToInsert);
        }
    }

    private void insertIntoQueue(int shard, List<CampaignToResync> campaignsToInsert) {
        var insertQuery = dslContextProvider.ppc(shard).insertQuery(BALANCE_INFO_QUEUE);
        for (CampaignToResync campaignToResync : campaignsToInsert) {
            Map<TableField<BalanceInfoQueueRecord, ?>, ?> fieldValues = Map.of(
                    BALANCE_INFO_QUEUE.OPERATOR_UID, campaignToResync.userId,
                    BALANCE_INFO_QUEUE.OBJ_TYPE, BalanceInfoQueueObjType.cid,
                    BALANCE_INFO_QUEUE.CID_OR_UID, campaignToResync.campaignId,
                    BALANCE_INFO_QUEUE.PRIORITY, PRIORITY_CAMP_WITH_BLOCKED_MONEY
            );
            insertQuery.addValues(fieldValues);
            insertQuery.newRecord();
        }
        insertQuery.execute();
    }
}
