package ru.yandex.direct.logicprocessor.processors.campstatusmoderate.handlers;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.InsertValuesStep2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbschema.ppc.tables.records.WarnPayRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.logicprocessor.processors.campstatusmoderate.CampaignStatus;

import static ru.yandex.direct.dbschema.ppc.Tables.WARN_PAY;

/**
 * Если кампания принята и не оплачена - делаем ей периодическую напоминалку о результате модерации
 * письма рассылает скрипт - moderateSendWarn.pl
 */
@Component
public class ModerationWarns implements CampaignStatusHandler {

    private static final Logger logger = LoggerFactory.getLogger(ModerationWarns.class);

    private final DslContextProvider dslContextProvider;
    private final CampaignRepository campaignRepository;

    @Autowired
    public ModerationWarns(DslContextProvider dslContextProvider, CampaignRepository campaignRepository) {
        this.dslContextProvider = dslContextProvider;
        this.campaignRepository = campaignRepository;
    }

    @Override
    public void process(int shard, Map<CampaignStatus, List<Long>> statusesToCids) {
        List<Long> forAdd = new ArrayList<>();
        List<Long> forRemove = new ArrayList<>();

        List<Long> allCids = statusesToCids.values().stream().flatMap(Collection::stream).collect(Collectors.toList());
        Map<Long, BigDecimal> moneyOnCampaigns = campaignRepository.getCampaignsSum(shard, allCids);

        for (var entry : statusesToCids.entrySet()) {
            for (var cid : entry.getValue()) {
                if (!moneyOnCampaigns.containsKey(cid)) {
                    continue; //Maybe was removed or whatever
                }

                if (moneyOnCampaigns.get(cid).compareTo(BigDecimal.ZERO) <= 0) {
                    CampaignStatusModerate newStatus = entry.getKey().getStatusModerate();
                    if (newStatus == CampaignStatusModerate.YES) {
                        forAdd.add(cid);
                    } else if (newStatus == CampaignStatusModerate.NO) {
                        forRemove.add(cid);
                    }
                }
            }
        }

        if (!forAdd.isEmpty()) {
            logger.debug("Sending payment reminders for campaigns: {}", forAdd);
            addWarn(shard, forAdd);
        }

        if (!forRemove.isEmpty()) {
            logger.debug("Remove payment reminders for campaigns: {}", forRemove);
            removeWarn(shard, forRemove);
        }
    }

    private void addWarn(int shard, Collection<Long> cids) {
        if (cids.isEmpty()) {
            return;
        }

        InsertValuesStep2<WarnPayRecord, Long, Long> insertStep = dslContextProvider.ppc(shard).insertInto(WARN_PAY)
                .columns(WARN_PAY.CID, WARN_PAY.WARNED_INTERVAL);

        for (Long cid : cids) {
            insertStep = insertStep.values(cid, 0L);
        }

        insertStep.onDuplicateKeyIgnore().execute();
    }

    private void removeWarn(int shard, Collection<Long> cids) {
        if (cids.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard).update(WARN_PAY)
                .set(WARN_PAY.WARNED_INTERVAL, 100L)
                .where(WARN_PAY.CID.in(cids))
                .execute();
    }
}
