package ru.yandex.direct.logicprocessor.processors.feeds.usagetypes

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.ess.config.feeds.usagetypes.FeedUsageTypesConfig
import ru.yandex.direct.ess.logicobjects.feeds.usagetypes.FeedUsageTypesObject
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.JugglerChecks
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext
import ru.yandex.direct.rbac.RbacService

private val LOGGER = LoggerFactory.getLogger(FeedUsageTypeProcessor::class.java)

@JugglerChecks(
    JugglerCheck(
        ttl = JugglerCheck.Duration(minutes = 5), needCheck = ProductionOnly::class,
        tags = [CheckTag.DIRECT_PRIORITY_0],
        notifications = [
            OnChangeNotification(
                recipient = [NotificationRecipient.LOGIN_BUHTER],
                status = [JugglerStatus.OK, JugglerStatus.CRIT],
                method = [NotificationMethod.TELEGRAM]
            )
        ]
    )
)
@EssLogicProcessor(FeedUsageTypesConfig::class)
class FeedUsageTypeProcessor(
    private val mbiService: MbiService,
    private val feedRepository: FeedRepository,
    private val rbacService: RbacService,
    val essLogicProcessorContext: EssLogicProcessorContext
) : BaseLogicProcessor<FeedUsageTypesObject>(essLogicProcessorContext) {

    override fun process(logicObjects: List<FeedUsageTypesObject>) {
        val feedIds = logicObjects.map { it.feedId }
        val feeds = feedRepository.getSimple(shard, feedIds)

        //Работаем только с теми фидами, для которых есть marketShopId
        val feedsWithMarketIds = feeds.filter { it.marketShopId != null }

        //Включаем шопы в MBI(повторное включение — не беда)
        setFeedStatusEnableInMBI(feedsWithMarketIds)

        //Проставляем обновленные фичи в MBI
        setFeedFeaturesInMBI(feedsWithMarketIds)
    }

    private fun setFeedFeaturesInMBI(feeds: List<FeedSimple>) {
        val relaxedWorker = RelaxedWorker()
        val marketIdToUsageTypes = feeds.associate { it.marketShopId to it.usageTypes }

        LOGGER.info(
            "Set features for shop in MBI for ${marketIdToUsageTypes.size} feeds on shard: $shard"
        )
        marketIdToUsageTypes.forEach {
            relaxedWorker.runAndRelax<RuntimeException> {
                mbiService.setFeedUsageType(
                    it.key,
                    it.value
                )
            }
        }
        LOGGER.info(
            "Done with setting features in MBI for ${marketIdToUsageTypes.size} feeds on shard: $shard"
        )
    }

    private fun setFeedStatusEnableInMBI(feeds: List<FeedSimple>) {
        val clientIds = feeds.map { ClientId.fromLong(it.clientId) }.toSet()
        val chiefUidsByClientId = rbacService.getChiefsByClientIds(clientIds)
        val relaxedWorker = RelaxedWorker()
        LOGGER.info(
            "Enable shop in MBI for ${feeds.size} feeds on shard: $shard"
        )
        feeds.forEach { feed ->
            val clientId = ClientId.fromLong(feed.clientId)
            relaxedWorker.runAndRelax<RuntimeException> {
                mbiService.sendFeed(
                    clientId,
                    chiefUidsByClientId[clientId]!!,
                    feed
                )
            }
        }
        LOGGER.info(
            "Done with enabling in MBI for ${feeds.size} feeds on shard: $shard"
        )
    }
}
