package ru.yandex.direct.logicprocessor.processors.feeds.usagetypes

import one.util.streamex.EntryStream
import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.binlog.model.Operation
import ru.yandex.direct.core.entity.feed.feedWithUsageTypeToAppliedChanges
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.model.FeedUsageType
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.feed.service.FeedUsageService
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.ess.config.feeds.usagetypes.RecalculateFeedUsageTypesConfig
import ru.yandex.direct.ess.logicobjects.feeds.usagetypes.FeedUsageTypesObject
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.JugglerChecks
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext
import ru.yandex.direct.utils.CollectionUtils.isEmpty
import ru.yandex.direct.utils.CollectionUtils.isNotEmpty
import java.util.EnumSet

@JugglerChecks(
    JugglerCheck(
        ttl = JugglerCheck.Duration(minutes = 5), needCheck = ProductionOnly::class,
        tags = [CheckTag.DIRECT_PRIORITY_0],
        notifications = [
            OnChangeNotification(
                recipient = [NotificationRecipient.LOGIN_BUHTER],
                status = [JugglerStatus.OK, JugglerStatus.CRIT],
                method = [NotificationMethod.TELEGRAM]
            )
        ]
    )
)
@EssLogicProcessor(RecalculateFeedUsageTypesConfig::class)
class RecalculateFeedUsageTypeProcessor(
    private val feedRepository: FeedRepository,
    private val feedService: FeedService,
    private val feedUsageService: FeedUsageService,
    val essLogicProcessorContext: EssLogicProcessorContext
) : BaseLogicProcessor<FeedUsageTypesObject>(essLogicProcessorContext) {
    companion object {
        private val LOGGER = LoggerFactory.getLogger(RecalculateFeedUsageTypeProcessor::class.java)
        private const val CHUNK_UPDATE_SIZE = 1000
    }

    override fun process(logicObjects: List<FeedUsageTypesObject>) {
        val feedsToUpdateUsage = getFeedsToNewUsage(logicObjects)

        // Обновляем статус фидов, для которых это требуется
        if (feedsToUpdateUsage.isEmpty()) {
            LOGGER.info("Got no feeds to update usage types on shard: $shard")
        } else {
            LOGGER.info("Got ${feedsToUpdateUsage.size} feeds to update usage types on shard: $shard")

            val appliedChanges = feedWithUsageTypeToAppliedChanges(feedsToUpdateUsage)

            appliedChanges
                .chunked(CHUNK_UPDATE_SIZE)
                .forEachIndexed { index, appliedChangesChunk ->
                    LOGGER.info(
                        "Updating ${index + 1} from ${appliedChanges.size / CHUNK_UPDATE_SIZE + 1} " +
                            "chunk feeds to update usage types on shard: $shard"
                    )
                    feedRepository.update(shard, appliedChangesChunk)
                }
        }
    }

    fun getFeedsToNewUsage(logicObjects: List<FeedUsageTypesObject>): Map<FeedSimple, EnumSet<FeedUsageType>> {
        val feedIdsInvolvedWithChangesCount = trackFeedUsageChanges(logicObjects)

        // Достаем из базы упрощенные вид фидов, использование которых могло поменяться
        val feedsInvolved = feedService.getFeedsSimple(shard, feedIdsInvolvedWithChangesCount.keys)

        // фильтруем очевидные кейсы: включенные, которые дополнительно только "включились",
        // выключенные, которые дополнительно только "выключились",
        // чтобы снизить кол-во фидов для которых все пересчитывать
        val feedIdsToRecalculateUsageTypes = feedsInvolved.filterNot { feed ->
            val changesCount = feedIdsInvolvedWithChangesCount[feed.id]!!
            usedFeedNotTurnedOff(feed, changesCount) || unusedFeedNotTurnedOn(feed, changesCount)
        }.map { it.id }

        // Считаем актуальный статус использования фидов, для тех случаев, когда он мог измениться
        val actualFeedUsageTypeByFeedId =
            feedUsageService.getActualFeedUsageTypeByFeedId(shard, feedIdsToRecalculateUsageTypes)

        return feedsInvolved
            .filter {
                actualFeedUsageTypeByFeedId.containsKey(it.id)
                    && actualFeedUsageTypeByFeedId[it.id] != it.usageTypes
            }
            .associateWith { actualFeedUsageTypeByFeedId[it.id]!! }
    }

    private fun trackFeedUsageChanges(logicObjects: List<FeedUsageTypesObject>): Map<Long, FeedUsageChangesCount> {
        // Выделяем разные виды изменений
        val campaignLogicObjectsByCampaignIds =
            logicObjects.filter { it.campaign_id != null }.associateBy { it.campaign_id!! }
        val adGroupsLogicObjectsByAdGroupIds =
            logicObjects.filter { it.adgroup_id != null }.associateBy { it.adgroup_id!! }
        val feedLogicObjectsByFeedIds = logicObjects.filter { it.feedId != null }.associateBy { it.feedId!! }

        // Получаем провязку изменений и фидов
        val feedIdsByCampaignIds: Map<Long, Set<Long>> =
            feedService.getFeedIdsByCampaignIds(shard, campaignLogicObjectsByCampaignIds.keys)
        val feedIdsByAdGroupIds: Map<Long, Long> =
            feedService.getFeedIdsByAdGroupIds(shard, adGroupsLogicObjectsByAdGroupIds.keys)

        val campaignIdsByFeedIds: Map<Long, List<Long>> = EntryStream.of(feedIdsByCampaignIds)
            .invert()
            .flatMapKeys { it.stream() }
            .grouping()

        val adGroupIdsByFeedIds: Map<Long, List<Long>> = EntryStream.of(feedIdsByAdGroupIds)
            .invert()
            .grouping()

        // Берем полный список id фидов, которые могли быть затронуты
        val feedIdsInvolved =
            feedLogicObjectsByFeedIds.keys + feedIdsByAdGroupIds.values + feedIdsByCampaignIds.values.flatten()

        LOGGER.info(
            "Got ${feedIdsInvolved.size} feeds(${feedLogicObjectsByFeedIds.size} feed changes, ${feedIdsByAdGroupIds.size} " +
                "adgroup changes, ${feedIdsByCampaignIds.size} campaigns changes) " +
                "involved to processing on shard $shard"
        )

        // Вычисляем виды изменений:
        // Если фид появился в добавлении групп — может быть только включение
        // Если фид появился в удалении — может быть только выключение
        // Если фид появился в обновлении — нельзя заранее сказать, что произошло
        val feedIdsInvolvedWithChangesCount = feedIdsInvolved.associateWith { FeedUsageChangesCount(0, 0, 0) }
        feedIdsInvolvedWithChangesCount.forEach { (feedId, changeCounter) ->
            if (feedLogicObjectsByFeedIds[feedId]?.operation == Operation.DELETE) {
                changeCounter.disabledCount++
            }
            if (feedLogicObjectsByFeedIds[feedId]?.operation == Operation.INSERT) {
                changeCounter.enabledCount++
            }

            val campaignIdsForFeed = campaignIdsByFeedIds[feedId] ?: emptyList()
            val campaignLogicObjectsForFeed =
                campaignLogicObjectsByCampaignIds.filter { entry -> campaignIdsForFeed.contains(entry.key) }
            changeCounter.disabledCount += campaignLogicObjectsForFeed
                .count { entry -> entry.value.operation == Operation.DELETE }
            changeCounter.unknownChangesCount +=
                campaignLogicObjectsForFeed.count { entry -> entry.value.operation != Operation.DELETE }

            val adGroupIdsForFeed = adGroupIdsByFeedIds[feedId] ?: emptyList()
            val adGroupLogicObjectsForFeed =
                adGroupsLogicObjectsByAdGroupIds.filter { entry -> adGroupIdsForFeed.contains(entry.key) }
            changeCounter.disabledCount += adGroupLogicObjectsForFeed
                .count { entry -> entry.value.operation == Operation.DELETE }
            changeCounter.unknownChangesCount += adGroupLogicObjectsForFeed
                .count { entry -> entry.value.operation != Operation.DELETE }
        }
        return feedIdsInvolvedWithChangesCount
    }

    private fun usedFeedNotTurnedOff(feed: FeedSimple, changesCount: FeedUsageChangesCount): Boolean =
        isNotEmpty(feed.usageTypes) && changesCount.disabledCount == 0 && changesCount.unknownChangesCount == 0

    private fun unusedFeedNotTurnedOn(feed: FeedSimple, changesCount: FeedUsageChangesCount): Boolean =
        isEmpty(feed.usageTypes) && changesCount.enabledCount == 0 && changesCount.unknownChangesCount == 0
}

data class FeedUsageChangesCount(
    var enabledCount: Int,
    var unknownChangesCount: Int,
    var disabledCount: Int
)
