package ru.yandex.direct.logicprocessor.processors.mediascopeintegration;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.service.MediascopePositionService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.mediascopeintegration.MediascopePositionChangeConfig;
import ru.yandex.direct.ess.logicobjects.mediascopeintegration.MediascopePositionChangeObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.mediascope.model.request.MediascopePosition;
import ru.yandex.direct.tracing.Trace;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

/**
 * Процессор для отправки позиций в Mediascope при изменениях в кампаниях и баннерах,
 * является частью интеграции с верификатором Mediascope,
 * синхронизирует наши баннеры с https://viewability.mediascope.net/
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_0})
@EssLogicProcessor(MediascopePositionChangeConfig.class)
@ParametersAreNonnullByDefault
public class MediascopePositionChangeProcessor extends BaseLogicProcessor<MediascopePositionChangeObject> {
    private static final Logger logger = LoggerFactory.getLogger(MediascopePositionChangeProcessor.class);
    private static final String TRACE_FUNC = "mediascope_integration:position_change_processor";

    private final MediascopePositionService mediascopePositionService;
    private final BannerCollectorService bannerCollectorService;

    protected MediascopePositionChangeProcessor(
            EssLogicProcessorContext essLogicProcessorContext,
            MediascopePositionService mediascopePositionService,
            BannerCollectorService bannerCollectorService) {
        super(essLogicProcessorContext);
        this.mediascopePositionService = mediascopePositionService;
        this.bannerCollectorService = bannerCollectorService;
    }

    @Override
    public void process(List<MediascopePositionChangeObject> logicObjects) {
        List<Long> bannerIdsWithIntegration;
        try (var ignore = Trace.current().profile(TRACE_FUNC, "collect_banners")) {
            bannerIdsWithIntegration = bannerCollectorService.getBannerIdsWithIntegration(getShard(), logicObjects);
        }

        if (bannerIdsWithIntegration.isEmpty()) {
            return;
        } else {
            logger.info("Collected {} banners", bannerIdsWithIntegration.size());
        }

        Map<Long, MediascopePosition> mediascopePositionsMap;
        try (var ignore = Trace.current().profile(TRACE_FUNC, "collect_positions")) {
            mediascopePositionsMap = mediascopePositionService.collectPositions(getShard(), bannerIdsWithIntegration);
        }

        try (var ignore = Trace.current().profile(TRACE_FUNC, "send_positions")) {
            mediascopePositionService.sendPositions(mediascopePositionsMap.values());
        }
    }
}
