package ru.yandex.direct.logicprocessor.processors.metrika.bannerupdate;

import java.time.Duration;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerProperties;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerPropertiesImpl;
import ru.yandex.direct.ess.config.metrika.bannerupdate.MetrikaBannerUpdateConfig;
import ru.yandex.direct.ess.logicobjects.metrika.bannerupdate.MetrikaBannerUpdateObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.metrika.bannerupdate.logbroker.MetrikaBannerUpdateLogbrokerWriter;
import ru.yandex.monlib.metrics.primitives.Counter;

import static ru.yandex.direct.ess.common.configuration.EssCommonConfiguration.ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_1})
@EssLogicProcessor(MetrikaBannerUpdateConfig.class)
public class MetrikaBannerUpdateProcessor extends BaseLogicProcessor<MetrikaBannerUpdateObject> {

    private static final Logger logger = LoggerFactory.getLogger(MetrikaBannerUpdateProcessor.class);
    private LogbrokerClientFactoryFacade logbrokerClientFactory;
    private LogbrokerWriter<MetrikaBannerUpdateResult> logbrokerWriter;

    private final MetrikaBannerUpdateService metrikaBannerUpdateService;
    private final DirectConfig logbrokerConfig;
    private Counter wrritenMassageCounter;

    @Autowired
    public MetrikaBannerUpdateProcessor(
            EssLogicProcessorContext essLogicProcessorContext,
            @Qualifier(ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME) LogbrokerClientFactoryFacade logbrokerClientFactory,
            MetrikaBannerUpdateService metrikaBannerUpdateService,
            DirectConfig logbrokerConfig) {
        super(essLogicProcessorContext);
        this.logbrokerClientFactory = logbrokerClientFactory;
        this.metrikaBannerUpdateService = metrikaBannerUpdateService;
        this.logbrokerConfig = logbrokerConfig.getBranch("metrika.bannerupdate.logbroker");
    }

    @Override
    protected void initialize() {
        int partitionsCount = logbrokerConfig.getInt("partitions_count");
        int group = getShard() % partitionsCount + 1;
        LogbrokerProducerProperties logbrokerProducerProperties = LogbrokerProducerPropertiesImpl.newBuilder()
                .setHost(logbrokerConfig.getString("host"))
                .setWriteTopic(logbrokerConfig.getString("topic"))
                .setTimeoutSec(logbrokerConfig.getDuration("writer_timeout").getSeconds())
                .setRetries(logbrokerConfig.getInt("retries"))
                .setGroup(group)
                .build();

        var producerSupplier =
                logbrokerClientFactory.createProducerSupplier(logbrokerProducerProperties,
                        String.valueOf(getShard()));

        logbrokerWriter = new MetrikaBannerUpdateLogbrokerWriter(producerSupplier,
                Duration.ofSeconds(logbrokerProducerProperties.getTimeoutSec()),
                logbrokerProducerProperties.getRetries());

        this.wrritenMassageCounter = getMetricRegistry().counter("written_metrika_objects");
    }

    @Override
    public void process(List<MetrikaBannerUpdateObject> logicObjects) {
        int shard = getShard();
        List<MetrikaBannerUpdateResult> resultsList =
                metrikaBannerUpdateService.getMetrikaBannerUpdateResultList(shard, logicObjects);
        int writingCnt = logbrokerWriter.writeSync(resultsList);
        wrritenMassageCounter.add(writingCnt);
        logger.info("Shard {}, written {} objects into LB", shard, writingCnt);
    }

    @Override
    protected void finishInitialized() {
        if (logbrokerWriter != null) {
            logbrokerWriter.close();
        }
    }
}
