package ru.yandex.direct.logicprocessor.processors.moderation;

import java.util.List;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.moderation.model.ModerationRequest;
import ru.yandex.direct.ess.logicobjects.moderation.AbstractModerationEvent;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.monlib.metrics.labels.Labels;

import static com.google.common.base.Preconditions.checkState;

public abstract class BaseModerationEventsProcessor<T extends AbstractModerationEvent> extends BaseLogicProcessor<T> {

    private static final Logger logger = LoggerFactory.getLogger(BaseModerationEventsProcessor.class);

    private List<? extends ModerationRequestsWriter> registeredWriters;

    public BaseModerationEventsProcessor(EssLogicProcessorContext essLogicProcessorContext) {
        super(essLogicProcessorContext);
        registeredWriters = List.of();
    }

    @Override
    protected void initialize() {
        registeredWriters = getWriters();
        checkState(!registeredWriters.isEmpty(), "No registered writers!");
        registeredWriters.forEach(this::initWriter);
    }

    private void initWriter(ModerationRequestsWriter<?> moderationRequestsWriter) {
        moderationRequestsWriter.initialize(getLogicProcessName(), getShard());
    }

    protected abstract List<? extends ModerationRequestsWriter> getWriters();

    @Override
    public final void process(List<T> logicObjects) {
        try (TraceProfile profile = Trace.current().profile("moderation_events_writer")) {
            processObjects(logicObjects);
        }
    }

    protected abstract void processObjects(List<T> events);

    @Override
    public void finishInitialized() {
        registeredWriters.forEach(ModerationRequestsWriter::finish);
        registeredWriters = List.of();
    }

    protected ModerationRequestsConsumerBuilder createModerationRequestsConsumerBuilder() {
        return new ModerationRequestsConsumerBuilder(this::getLogicProcessName, this::getShard);
    }

    // вот это конечно должно быть не здесь
    protected <R extends ModerationRequest> Consumer<List<R>> makeRequestsConsumer(
            ModerationRequestsWriter<R> writer,
            ModerationWriterMonitoring moderationWriterMonitoring,
            Labels labels) {
        return createModerationRequestsConsumerBuilder().build(
                writer,
                moderationWriterMonitoring,
                labels);
    }

    protected <R extends ModerationRequest> Consumer<List<R>> makeRequestsConsumer(
            ModerationRequestsWriter<R> writer,
            ModerationWriterMonitoring moderationWriterMonitoring) {
        return makeRequestsConsumer(writer, moderationWriterMonitoring, Labels.empty());
    }

    protected Labels getLabelsForMonitoring(Labels subLabels) {
        return createModerationRequestsConsumerBuilder().getLabelsForMonitoring(subLabels);
    }
}
