package ru.yandex.direct.logicprocessor.processors.moderation;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.ModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.Moderationable;
import ru.yandex.direct.core.entity.moderation.service.sending.ModerationSendingService;
import ru.yandex.direct.ess.logicobjects.moderation.BaseModerationObject;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEssTagMap;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEventTimeMap;

public abstract class SimpleModerationEventsProcessor<
        T extends BaseModerationObject,
        M extends Moderationable,
        D extends ModerationRequest<? extends ModerationMeta>> extends BaseModerationEventsProcessor<T> {
    private final Function<M, Long> keyExtractor;
    private final ModerationSendingService<Long, D, M> sender;

    private final ModerationRequestsWriter<D> moderationWriter;

    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    ;

    protected SimpleModerationEventsProcessor(EssLogicProcessorContext essLogicProcessorContext,
                                              Function<M, Long> keyExtractor,
                                              ModerationSendingService<Long, D, M> sender,
                                              ModerationRequestsWriter<D> moderationWriter,
                                              ModerationWriterMonitoring moderationWriterMonitoring,
                                              ModerationOperationModeProvider moderationOperationModeProvider) {
        super(essLogicProcessorContext);
        this.keyExtractor = keyExtractor;
        this.sender = sender;
        this.moderationWriter = moderationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    protected List<Long> getObjectIds(List<T> events) {
        return events.stream().map(BaseModerationObject::getObjectId).collect(Collectors.toList());
    }

    @Override
    protected List<ModerationRequestsWriter> getWriters() {
        return List.of(moderationWriter);
    }


    @Override
    protected void processObjects(List<T> events) {
        Map<Boolean, List<T>> grouppedByCopyStatus =
                events.stream()
                        .collect(Collectors.groupingBy(BaseModerationObject::getWasCopied, Collectors.toList()));

        //Order is important, because each of  next calls makes its own read from the database, so only the last call
        // gets
        // the latest object version.
        handleCopiedObjects(grouppedByCopyStatus.getOrDefault(Boolean.TRUE, List.of()));
        sendObjectsToModeration(grouppedByCopyStatus.getOrDefault(Boolean.FALSE, List.of()));
    }

    private void handleCopiedObjects(List<T> events) {
        if (events.isEmpty()) {
            return;
        }

        try {
            this.moderationOperationModeProvider.forceMode(RESTRICTED);
            sendObjectsToModeration(events);
        } finally {
            this.moderationOperationModeProvider.disableForcedMode();
        }
    }

    private void sendObjectsToModeration(List<T> events) {
        if (events.isEmpty()) {
            return;
        }
        try (TraceProfile profile = Trace.current().profile(
                sender.typeName() + "_moderation_requests.make", String.valueOf(getShard()))) {

            List<Long> objectIds = getObjectIds(events);

            if (!objectIds.isEmpty()) {
                Map<Long, Long> objectIdToEventTimeMap =
                        getIdToEventTimeMap(events, BaseModerationObject::getObjectId);

                Map<Long, String> objectIdToEssTagMap =
                        getIdToEssTagMap(events, BaseModerationObject::getObjectId);

                sender.send(getShard(), objectIds,
                        (o) -> objectIdToEventTimeMap.get(keyExtractor.apply(o)),
                        (o) -> objectIdToEssTagMap.get(keyExtractor.apply(o)),
                        makeRequestsConsumer(moderationWriter, moderationWriterMonitoring,
                                getTypeSpecificLabelsForMonitoring()));
            }
        }
    }

    protected abstract Labels getTypeSpecificLabelsForMonitoring();

}
