package ru.yandex.direct.logicprocessor.processors.moderation.adgroup;

import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.moderation.ModerationOperationMode;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.adgroup.AdGroupModerationRequest;
import ru.yandex.direct.core.entity.moderation.service.sending.adgroup.AdGroupSender;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.adgroup.AdGroupModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.adgroup.AdGroupModerationEventObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BaseModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.AdGroupModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.monlib.metrics.labels.Labels;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_ADGROUP_TRANSPORT_NEW_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEssTagMap;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEventTimeMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_0, DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags = {DIRECT_PRIORITY_0})
@EssLogicProcessor(AdGroupModerationConfig.class)
public class AdGroupModerationEventsProcessor extends BaseModerationEventsProcessor<AdGroupModerationEventObject> {

    private final AdGroupSender adGroupSender;
    private final AdGroupRepository adGroupRepository;
    private final AdGroupModerationWriter moderationWriter;
    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;

    private final PpcProperty<Integer> adGroupTransportEnabledProp;

    @Autowired
    public AdGroupModerationEventsProcessor(
            EssLogicProcessorContext essLogicProcessorContext,
            AdGroupSender adGroupSender,
            AdGroupRepository adGroupRepository,
            AdGroupModerationWriter moderationWriter,
            ModerationWriterMonitoring moderationWriterMonitoring,
            ModerationOperationModeProvider moderationOperationModeProvider,
            PpcPropertiesSupport ppcPropertiesSupport) {

        super(essLogicProcessorContext);

        this.adGroupSender = adGroupSender;
        this.adGroupRepository = adGroupRepository;
        this.moderationWriter = moderationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;

        this.adGroupTransportEnabledProp =
                ppcPropertiesSupport.get(ENABLE_ADGROUP_TRANSPORT_NEW_MODERATION, Duration.ofSeconds(15));
    }

    @Override
    protected void processObjects(List<AdGroupModerationEventObject> events) {
        try (TraceProfile profile = Trace.current().profile(
                "adgroup_moderation_requests.make", String.valueOf(getShard()))) {

            Map<Boolean, List<AdGroupModerationEventObject>> eventsByRestrictedMode =
                    StreamEx.of(events)
                            .mapToEntry(AdGroupModerationEventObject::getWasCopied, Function.identity())
                            .grouping();

            processObjectsInRestrictedMode(eventsByRestrictedMode.getOrDefault(Boolean.TRUE, emptyList()));
            processObjectsInternal(eventsByRestrictedMode.getOrDefault(Boolean.FALSE, emptyList()));
        }
    }

    private void processObjectsInRestrictedMode(List<AdGroupModerationEventObject> events) {
        try {
            moderationOperationModeProvider.forceMode(ModerationOperationMode.RESTRICTED);
            processObjectsInternal(events);
        } finally {
            moderationOperationModeProvider.disableForcedMode();
        }
    }

    private void processObjectsInternal(List<AdGroupModerationEventObject> events) {
        List<Long> adGroupIds = getAdGroupIdsForProcessing(events);
        if (adGroupIds.isEmpty()) {
            return;
        }

        Map<Long, Long> adGroupIdToEventTimeMap =
                getIdToEventTimeMap(events, AdGroupModerationEventObject::getAdGroupId);
        Map<Long, String> adGroupIdToEssTagMap =
                getIdToEssTagMap(events, AdGroupModerationEventObject::getAdGroupId);

        adGroupSender.send(getShard(), adGroupIds,
                adGroupWithModerationInfo -> adGroupIdToEventTimeMap.get(adGroupWithModerationInfo.getId()),
                adGroupWithModerationInfo -> adGroupIdToEssTagMap.get(adGroupWithModerationInfo.getId()),
                makeRequestsConsumer(moderationWriter, moderationWriterMonitoring,
                        Labels.of("type", "adgroup")));
    }

    @Override
    protected List<ModerationRequestsWriter<? extends AdGroupModerationRequest>> getWriters() {
        return List.of(moderationWriter);
    }

    private List<Long> getAdGroupIdsForProcessing(List<AdGroupModerationEventObject> events) {
        List<Long> adGroupIds = mapList(events, AdGroupModerationEventObject::getAdGroupId);
        Map<Long, ClientId> adGroupIdToClientIdMap =
                adGroupRepository.getClientIdsByAdGroupIds(getShard(), new HashSet<>(adGroupIds));
        return StreamEx.of(adGroupIds)
                .filter(adGroupId -> isClientProcessingEnabled(adGroupIdToClientIdMap.get(adGroupId)))
                .toList();
    }

    private boolean isClientProcessingEnabled(@Nullable ClientId clientId) {
        return clientId != null && clientId.asLong() % 100 < adGroupTransportEnabledProp.getOrDefault(0);
    }
}
